/*** events.h -- New Generation Events
 *
 * Copyright (C) 2006, 2007, 2008 Sebastian Freundt
 *
 * Author:  Sebastian Freundt <hroptatyr@sxemacs.org>
 *
 * This file is part of SXEmacs.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * 3. Neither the name of the author nor the names of any contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR "AS IS" AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
 * IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ***/

/* Inspired by XEmacs' events.c written by Jamie Zawinski */

#ifndef INCLUDED_events_h_
#define INCLUDED_events_h_

#include <kant/kant.h>
#if defined HAVE_METRONOME && 0
# include "metronome.h"
#endif
#ifdef HAVE_THREADS
# include "semaphore.h"
#endif

#ifdef ALL_DEBUG_FLAGS
#undef EVENTS_DEBUG_FLAG
#define EVENTS_DEBUG_FLAG
#endif

#define __EVENTS_DEBUG__(args...)	fprintf(stderr, "events " args)
#ifndef EVENTS_DEBUG_FLAG
#define EVENTS_DEBUG(args...)
#else
#define EVENTS_DEBUG(args...)		__EVENTS_DEBUG__(args)
#endif
#define EVENTS_DEBUG_EVENT(args...)	EVENTS_DEBUG("[event] " args)
#define EVENTS_DEBUG_EVHDL(args...)	EVENTS_DEBUG("[event-handler] " args)
#define EVENTS_CRITICAL(args...)	__EVENTS_DEBUG__("CRITICAL: " args)

typedef struct event_s *event_t;
typedef struct event_handler_s *event_handler_t;

typedef void(*evhdl_prnt_f)(event_t);
typedef void(*evhdl_handle_f)(event_t);
typedef evhdl_handle_f(*evhdl_treat_f)(event_t);
typedef void(*evhdl_free_f)(event_t);
typedef event_t(*evhdl_getev_f)(event_t);
typedef void(*evhdl_putev_f)(event_t, event_t);
typedef handle_t(*evhdl_gethd_f)(event_t);
typedef void(*evhdl_puthd_f)(event_t, handle_t);

extern_inline void *get_event_data(event_t ev);
extern_inline void set_event_data(event_t ev, void *data);
extern_inline void *get_event_result(event_t ev);
extern_inline void set_event_result(event_t ev, void *res);
/* getter only */
extern_inline void *get_event_timestamp(event_t ev);
extern_inline evhdl_prnt_f get_evhdl_prnt_f(event_handler_t hdl);
extern_inline evhdl_prnt_f get_event_prnt_f(event_t ev);
extern_inline void set_evhdl_prnt_f(event_handler_t hdl, evhdl_prnt_f pf);
extern_inline evhdl_handle_f get_evhdl_handle_f(event_handler_t hdl);
extern_inline evhdl_handle_f get_event_handle_f(event_t ev);
extern_inline void set_evhdl_handle_f(event_handler_t hdl, evhdl_handle_f hf);
extern_inline evhdl_free_f get_evhdl_free_f(event_handler_t hdl);
extern_inline evhdl_free_f get_event_free_f(event_t ev);
extern_inline void set_evhdl_free_f(event_handler_t hdl, evhdl_free_f ff);
extern_inline evhdl_getev_f get_evhdl_getev_f(event_handler_t hdl);
extern_inline evhdl_getev_f get_event_getev_f(event_t ev);
extern_inline void set_evhdl_getev_f(event_handler_t hdl, evhdl_getev_f pf);
extern_inline evhdl_putev_f get_evhdl_putev_f(event_handler_t hdl);
extern_inline evhdl_putev_f get_event_putev_f(event_t ev);
extern_inline void set_evhdl_putev_f(event_handler_t hdl, evhdl_putev_f pf);
extern_inline evhdl_gethd_f get_evhdl_gethd_f(event_handler_t hdl);
extern_inline evhdl_gethd_f get_event_gethd_f(event_t ev);
extern_inline void set_evhdl_gethd_f(event_handler_t hdl, evhdl_gethd_f pf);
extern_inline evhdl_puthd_f get_evhdl_puthd_f(event_handler_t hdl);
extern_inline evhdl_puthd_f get_event_puthd_f(event_t ev);
extern_inline void set_evhdl_puthd_f(event_handler_t hdl, evhdl_puthd_f pf);
extern_inline void *get_evhdl_subhandler(event_handler_t hdl);
extern_inline void *get_event_subhandler(event_t ev);
extern_inline void set_evhdl_subhandler(event_handler_t hdl, void *handler);
extern_inline void set_event_subhandler(event_t ev, void *handler);
extern_inline void clear_evhdl_subhandler(event_handler_t hdl);
extern_inline void clear_event_subhandler(event_t ev);
extern_inline char *get_event_buffer(const event_t ev);
extern_inline size_t get_event_buffer_size(const event_t ev);
extern_inline void
obtain_event_buffer(char **buf, size_t *size, const event_t ev);
extern_inline void event_lock(event_t ev);
extern_inline void event_unlock(event_t ev);


struct event_s {
	event_handler_t handler;
#if defined HAVE_METRONOME && 0
	metronome_t timestamp;
#else
	long unsigned int timestamp;
#endif
	void *data;		/* custom event data */
	long int flags;		/* who knows, they become useful one day */
	sxe_mutex_t mtx;

	void *result;

	/* some support from the underlying worker */
	void *buffer;
	size_t buffer_size;
};

struct event_handler_s {
	evhdl_prnt_f prnt_f;
	evhdl_handle_f handle_f;
	evhdl_free_f free_f;
	evhdl_putev_f putev_f;
	evhdl_getev_f getev_f;
	evhdl_puthd_f puthd_f;
	evhdl_gethd_f gethd_f;
	void *subhandler;	/* a very custom handler */
};

#define event_handler(a)	((a)->handler)
#define event_timestamp(a)	((a)->timestamp)
#define event_data(a)		((a)->data)
#define event_flags(a)		((a)->flags)
#define event_mtx(a)		((a)->mtx)
#define event_result(a)		((a)->result)
#define event_buffer(a)		((a)->buffer)
#define event_buffer_size(a)	((a)->buffer_size)

#define evhdl_prnt_f(a)		((a)->prnt_f)
#define evhdl_handle_f(a)	((a)->handle_f)
#define evhdl_free_f(a)		((a)->free_f)
#define evhdl_putev_f(a)	((a)->putev_f)
#define evhdl_getev_f(a)	((a)->getev_f)
#define evhdl_puthd_f(a)	((a)->puthd_f)
#define evhdl_gethd_f(a)	((a)->gethd_f)
#define evhdl_subhandler(a)	((a)->subhandler)

/* more general stuff */
extern event_t make_event(event_handler_t);
extern void free_event(event_t);
extern void print_event(event_t);
extern void handle_event(event_t);
extern void log_event(event_t);

/* some funs (inlines) for the event handler */
extern_inline void*
get_event_data(event_t ev)
{
	return event_data(ev);
}

extern_inline void
set_event_data(event_t ev, void *data)
{
	event_data(ev) = data;
	return;
}

extern_inline void*
get_event_result(event_t ev)
{
	return event_result(ev);
}

extern_inline void
set_event_result(event_t ev, void *res)
{
	event_result(ev) = res;
	return;
}

/* getter only */
extern_inline void*
get_event_timestamp(event_t ev)
{
	return (void*)event_timestamp(ev);
}

extern_inline evhdl_prnt_f
get_evhdl_prnt_f(event_handler_t hdl)
{
	return evhdl_prnt_f(hdl);
}

extern_inline evhdl_prnt_f
get_event_prnt_f(event_t ev)
{
	return get_evhdl_prnt_f(event_handler(ev));
}

extern_inline void
set_evhdl_prnt_f(event_handler_t hdl, evhdl_prnt_f pf)
{
	evhdl_prnt_f(hdl) = pf;
	return;
}

extern_inline evhdl_handle_f
get_evhdl_handle_f(event_handler_t hdl)
{
	return evhdl_handle_f(hdl);
}

extern_inline evhdl_handle_f
get_event_handle_f(event_t ev)
{
	return get_evhdl_handle_f(event_handler(ev));
}

extern_inline void
set_evhdl_handle_f(event_handler_t hdl, evhdl_handle_f hf)
{
	evhdl_handle_f(hdl) = hf;
	return;
}

extern_inline evhdl_free_f
get_evhdl_free_f(event_handler_t hdl)
{
	return evhdl_free_f(hdl);
}

extern_inline evhdl_free_f
get_event_free_f(event_t ev)
{
	return get_evhdl_free_f(event_handler(ev));
}

extern_inline void
set_evhdl_free_f(event_handler_t hdl, evhdl_free_f ff)
{
	evhdl_free_f(hdl) = ff;
	return;
}

extern_inline evhdl_getev_f
get_evhdl_getev_f(event_handler_t hdl)
{
	return evhdl_getev_f(hdl);
}

extern_inline evhdl_getev_f
get_event_getev_f(event_t ev)
{
	return get_evhdl_getev_f(event_handler(ev));
}

extern_inline void
set_evhdl_getev_f(event_handler_t hdl, evhdl_getev_f pf)
{
	evhdl_getev_f(hdl) = pf;
	return;
}

extern_inline evhdl_putev_f
get_evhdl_putev_f(event_handler_t hdl)
{
	return evhdl_putev_f(hdl);
}

extern_inline evhdl_putev_f
get_event_putev_f(event_t ev)
{
	return get_evhdl_putev_f(event_handler(ev));
}

extern_inline void
set_evhdl_putev_f(event_handler_t hdl, evhdl_putev_f pf)
{
	evhdl_putev_f(hdl) = pf;
	return;
}

extern_inline evhdl_gethd_f
get_evhdl_gethd_f(event_handler_t hdl)
{
	return evhdl_gethd_f(hdl);
}

extern_inline evhdl_gethd_f
get_event_gethd_f(event_t ev)
{
	return get_evhdl_gethd_f(event_handler(ev));
}

extern_inline void
set_evhdl_gethd_f(event_handler_t hdl, evhdl_gethd_f pf)
{
	evhdl_gethd_f(hdl) = pf;
	return;
}

extern_inline evhdl_puthd_f
get_evhdl_puthd_f(event_handler_t hdl)
{
	return evhdl_puthd_f(hdl);
}

extern_inline evhdl_puthd_f
get_event_puthd_f(event_t ev)
{
	return get_evhdl_puthd_f(event_handler(ev));
}

extern_inline void
set_evhdl_puthd_f(event_handler_t hdl, evhdl_puthd_f pf)
{
	evhdl_puthd_f(hdl) = pf;
	return;
}

extern_inline void*
get_evhdl_subhandler(event_handler_t hdl)
{
	return evhdl_subhandler(hdl);
}

extern_inline void*
get_event_subhandler(event_t ev)
{
	return get_evhdl_subhandler(event_handler(ev));
}

extern_inline void
set_evhdl_subhandler(event_handler_t hdl, void *handler)
{
	evhdl_subhandler(hdl) = handler;
	return;
}

extern_inline void
set_event_subhandler(event_t ev, void *handler)
{
	set_evhdl_subhandler(event_handler(ev), handler);
	return;
}

extern_inline void
clear_evhdl_subhandler(event_handler_t hdl)
{
	set_evhdl_subhandler(hdl, NULL);
	return;
}

extern_inline void
clear_event_subhandler(event_t ev)
{
	clear_evhdl_subhandler(event_handler(ev));
	return;
}

extern_inline char*
get_event_buffer(const event_t ev)
{
	return event_buffer(ev);
}

extern_inline size_t
get_event_buffer_size(const event_t ev)
{
	return event_buffer_size(ev);
}

extern_inline void
obtain_event_buffer(char **buf, size_t *size, const event_t ev)
{
	*buf = event_buffer(ev);
	*size = event_buffer_size(ev);
}

extern_inline void
event_lock(event_t ev)
{
	SXE_MUTEX_LOCK(&event_mtx(ev));
}

extern_inline void
event_unlock(event_t ev)
{
	SXE_MUTEX_UNLOCK(&event_mtx(ev));
}

extern volatile int sigint_happened;

extern void init_events(void);
extern void reinit_events(void);
extern void deinit_events(void);

#endif	/* INCLUDED_events_h_ */
