/*** inputs.h -- Various Input Sources
 *
 * Copyright (C) 2006, 2007, 2008 Sebastian Freundt
 *
 * Author:  Sebastian Freundt <hroptatyr@sxemacs.org>
 *
 * This file is part of SXEmacs.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * 3. Neither the name of the author nor the names of any contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR "AS IS" AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
 * IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ***/

#ifndef INCLUDED_inputs_h
#define INCLUDED_inputs_h

#include "events.h"
#include "protos.h"
#if defined HAVE_POPT && defined WITH_POPT
# include <popt.h>
#endif	/* HAVE_POPT && WITH_POPT */

#ifdef ALL_DEBUG_FLAGS
#undef INPUT_DEBUG_FLAG
#define INPUT_DEBUG_FLAG		1
#endif

#define __INPUT_DEBUG__(args...)	fprintf(stderr, "INPUT " args)
#ifdef INPUT_DEBUG_FLAG
#define INPUT_DEBUG(args...)		__INPUT_DEBUG__(args)
#else
#define INPUT_DEBUG(args...)
#endif
#define INPUT_CRITICAL(args...)		__INPUT_DEBUG__("CRITICAL: " args)

extern event_handler_t Vevhdl_input;
extern event_handler_t Vevhdl_output;

typedef struct input_funs_s *input_funs_t;
typedef struct input_ctx_s *input_ctx_t;
typedef struct input_data_s *input_data_t;
typedef struct input_handler_s *input_handler_t;
typedef int(*inhdl_input_f)(event_t);
typedef int(*inhdl_output_f)(event_t);
typedef int(*inhdl_vanish_f)(event_t);
typedef void(*inhdl_free_f)(event_t);
typedef void(*inhdl_prnt_f)(event_t);

typedef int(*input_read_f)(input_ctx_t, register char*, size_t);
typedef int(*input_write_f)(input_ctx_t, const char*, size_t);
typedef int(*input_close_f)(input_ctx_t);
typedef void(*input_watch_f)(input_ctx_t);
typedef void(*input_ignore_f)(input_ctx_t);
typedef void(*input_suicide_f)(input_ctx_t);
typedef input_data_t(*input_data_f)(input_ctx_t);

extern_inline void *get_input_data(event_t inev);
extern_inline void set_input_data(event_t inev, void *data);
extern_inline void set_inhdl_prnt_f(input_handler_t hdl, inhdl_prnt_f);
extern_inline void set_inhdl_free_f(input_handler_t hdl, inhdl_free_f);
extern_inline void set_inhdl_input_f(input_handler_t hdl, inhdl_input_f);
extern_inline void set_inhdl_output_f(input_handler_t hdl, inhdl_output_f);


/* forward */
struct input_funs_s {
	input_read_f readf;
	input_write_f writef;
	input_close_f closef;
	input_watch_f watchf;
	input_ignore_f ignoref;
	input_suicide_f suicidef;
	input_data_f dataf;
};

struct input_ctx_s {
	int input;
	input_funs_t funs;

	int buf_len;
	char buffer[4096];
	void *proto;
	sxe_mutex_t mtx;
};

struct input_data_s {
	void *ctx;
	event_t inev;
	/* buffer */
	char *buffer;
	size_t size;
};

struct input_handler_s {
	inhdl_input_f input_f;
	inhdl_output_f output_f;
	inhdl_vanish_f vanish_f;
	inhdl_free_f free_f;
	inhdl_prnt_f prnt_f;
};

#define inhdl_input_f(s)	((s)->input_f)
#define inhdl_output_f(s)	((s)->output_f)
#define inhdl_vanish_f(s)	((s)->vanish_f)
#define inhdl_free_f(s)		((s)->free_f)
#define inhdl_prnt_f(s)		((s)->prnt_f)

/* inlines */
extern_inline void*
get_input_data(event_t inev)
{
	return get_event_data(inev);
}

extern_inline void
set_input_data(event_t inev, void *data)
{
	set_event_data(inev, data);
	return;
}

#if defined HAVE_POPT && defined WITH_POPT
extern struct poptOption input_opts[];
#endif
extern void init_inputs(void);
extern void reinit_inputs(void);
extern void deinit_inputs(void);

extern_inline void
set_inhdl_prnt_f(input_handler_t hdl, inhdl_prnt_f prntf)
{
	inhdl_prnt_f(hdl) = prntf;
	return;
}

extern_inline void
set_inhdl_free_f(input_handler_t hdl, inhdl_free_f freef)
{
	inhdl_free_f(hdl) = freef;
	return;
}

extern_inline void
set_inhdl_input_f(input_handler_t hdl, inhdl_input_f inputf)
{
	inhdl_input_f(hdl) = inputf;
	return;
}

extern_inline void
set_inhdl_output_f(input_handler_t hdl, inhdl_output_f outputf)
{
	inhdl_output_f(hdl) = outputf;
	return;
}

extern event_t make_input_event(void);
extern event_t make_output_event(event_t ev);


#ifdef WITH_INPUT_TCPUDP
#include "input-tcpudp.h"
#endif

#endif	/* INCLUDED_inputs_h */
