/*** lang-om.h -- Openmath support
 *
 * Copyright (C) 2006, 2007, 2008 Sebastian Freundt
 *
 * Author:  Sebastian Freundt <hroptatyr@sxemacs.org>
 *
 * This file is part of SXEmacs.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * 3. Neither the name of the author nor the names of any contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR "AS IS" AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
 * IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ***/

#ifndef INCLUDED_lang_om_h
#define INCLUDED_lang_om_h

#include "lang-xml.h"
#include <libxml/xpath.h>
#include <libxml/xpathInternals.h>
#include <libxml/xpointer.h>

/**
 * \defgroup lang_om The Openmath Language
 * \ingroup langs
 *
 * \brief Openmath specific stuff
 *
 * \{ */

/**
 * macro to output debugging information related to openmath features */
#define LANG_DEBUG_OM(args...)		LANG_DEBUG("[openmath] " args)
/**
 * macro to output debugging information when openmath XPath expressions
 * are concerned */
#define LANG_DEBUG_OM_EXPR(args...)		\
	LANG_DEBUG_OM("[expr] " args)
/**
 * macro to output critical messages related to openmath features */
#define LANG_CRITICAL_OM(args...)		\
	LANG_CRITICAL("[openmath] " args)

/**
 * A pointer to a language context. ..
 * The name protx is archaic though, we used to call the OM layer protocol X. */
typedef struct om_ctx_s *om_ctx_t;

/* hereafter a number of constructors, we use the naming scheme
 * lang_om_make_blabla() to just create an object, and
 * lang_om_blabla(...) for the combined construction and initial filling. */
/**
 * Return a newly created husk for an application (an OMA cell).
 *
 * \return a \c xmlNodePtr cast to \c void* */
extern_inline void *lang_om_make_application(void);
/**
 * Return a newly created husk for a symbol (an OMS cell).
 * 
 * \return a \c xmlNodePtr cast to \c void* */
extern_inline void *lang_om_make_symbol(void);
/**
 * Return a symbol cell (OMS) whose name is \p NAME in the cd \p CD.
 * 
 * \return a \c xmlNodePtr cast to \c void* */
extern_inline void *lang_om_symbol(const char *cd, const char *name);
/**
 * Dispatch a symbol cell (OMS) with name \p N in cd \p CD into the stream \p S.
 * 
 * \return a \c xmlNodePtr cast to \c void* */
extern_inline void *lang_om_disp_symbol(void *s, const char *cd, const char *n);
/**
 * Return an application cell (OMA) applied to a symbol cell (OMS).
 * whose name is \p NAME in the cd \p CD.
 * 
 * \return a \c xmlNodePtr cast to \c void* */
extern_inline void *lang_om_application(const char *cd, const char *name);
/**
 * Return a newly created husk for a binding (an OMBIND cell).
 * 
 * \return a \c xmlNodePtr cast to \c void* */
extern_inline void *lang_om_make_binding(void);
/**
 * Whatever. */
extern_inline void *lang_om_binding(const char *cd, const char *name);
/**
 * Return a newly created husk for a string (an OMSTR cell).
 * 
 * \return a \c xmlNodePtr cast to \c void* */
extern_inline void *lang_om_make_string(void);
/**
 * Return a string cell (OMSTR) from the contents of \p STRING of length \p LEN.
 * 
 * \return a \c xmlNodePtr cast to \c void* */
extern_inline void *lang_om_string(const char *string, size_t len);
/**
 * Return a newly created husk for a variable binding (an OMBVAR cell)
 * 
 * \return a \c xmlNodePtr cast to \c void* */
extern_inline void *lang_om_make_ombvar(void);
/**
 * Return a newly created husk for a variable symbol (an OMV cell).
 * 
 * \return a \c xmlNodePtr cast to \c void* */
extern_inline void *lang_om_make_varsym(void);
/**
 * Return a variable symbol cell (OMV) with name VARSYM.
 * 
 * \return a \c xmlNodePtr cast to \c void* */
extern_inline void *lang_om_varsym(const char *varsym);
/**
 * Return a newly created husk for an integer (an OMI cell).
 *
 * \return a \c xmlNodePtr cast to \c void* */
extern_inline void *lang_om_make_omi(void);
/**
 * Return an integer cell (OMI) from \p VALUE.
 * Note: This function is not aware of big integers.
 *
 * \param value the integer value to convert
 * \return a \c xmlNodePtr cast to \c void* */
extern_inline void *lang_om_omi(long int value);

/**
 * Return a variable binding, that is a combined OMBVAR/OMV cell from \p VARSYM.
 *
 * \param varsym the name of the variable symbol 
 * \return a \c xmlNodePtr cast to \c void* */
extern_inline void *lang_om_bind_to(const char *varsym);
/**
 * Return a reference to a named variable, an OMR cell, from \p VARSYM
 *
 * \param varsym the name of the variable symbol 
 * \return a \c xmlNodePtr cast to \c void* */
extern_inline void *lang_om_make_reference(const char *varsym);
/**
 * Return a newly created husk for an error (an OME cell).
 *
 * \return a \c xmlNodePtr cast to \c void* */
extern_inline void *lang_om_make_error(void);
/**
 * Return an error cell (OME) from \p ERRMSG of length \p LEN.
 *
 * \return a \c xmlNodePtr cast to \c void* */
extern_inline void *lang_om_error_string(const char *errmsg, size_t len);
/**
 * Return a symbol cell (OMS) named \p NAME in the cd \p CD.
 *
 * \param cd the name of the cd the symbol lies in
 * \param name the name of the symbol
 * \return a \c xmlNodePtr cast to \c void* */
extern_inline void *lang_om_error(const char *cd, const char *name);
/**
 * Return an OME cell from the scscp1.error_runtime symbol.
 * This cell in turn is supposed to be enriched with an OMSTR tag with a
 * verbose description of the error that was thrown.
 *
 * \return a \c xmlNodePtr cast to \c void* */
extern_inline void *lang_om_error_runtime(void);
/**
 * Return an OME cell over the error.unhandled_symbol symbol.
 * If \p SYM is non-\c NULL it should be a node pointer to the symbol that
 * cannot be handled and hence triggered this error handler
 *
 * \return a \c xmlNodePtr cast to \c void* */
extern_inline void *lang_om_error_unhandled_symbol(void *sym);
/**
 * Return an OME cell over the error.unexpected_symbol symbol.
 * If \p SYM is non-\c NULL it should be a node pointer to the symbol that
 * was unexpected and hence triggered this error handler
 *
 * \return a \c xmlNodePtr cast to \c void* */
extern_inline void *lang_om_error_unexpected_symbol(void *sym);
/**
 * Return an OME cell over the error.unsupported_CD symbol.
 * If \p SYM is non-\c NULL it should be a node pointer to the symbol
 * (as in OMS) whose cd attribute contains the unsupported dictionary
 * and hence triggered this error handler
 *
 * \return a \c xmlNodePtr cast to \c void* */
extern_inline void *lang_om_error_unsupported_CD(void *sym);
/**
 * Return \p NODE wrapped in an OMATTR attribution.
 * As of Openmath V2.0 we can attach attributes to every node along the path,
 * if optional argument \p ATP is non-\c NULL it will be filled
 * with the pointer to the ATP group of the attribution.
 *
 * \param atp a pointer to a pointer which is set to the OMATP cell.
 * \param node the node to be attribeautified, i.e. wrapped in an OMATTR cell.
 * \return a \c xmlNodePtr cast to \c void* */
extern_inline void*
lang_om_attribeautify_node(const void *restrict *atp, void *node);
/**
 * Return the content of the current node in \p CTX.
 * This is most likely a string, but also works otherwise in which case
 * you have to be extremely careful of what you do with the result.
 *
 * \param ctx the language context.
 * \return a \c void* which may be cast to a \c char* */
extern_inline void *lang_om_get_node_content(lang_ctx_t ctx);
/**
 * Return the current node of \p CTX.
 *
 * \param ctx the language context.
 * \return a \c xmlNodePtr cast to \c void* */
extern_inline void *lang_om_get_node(lang_ctx_t ctx);
/**
 * Set by side-effect the current node of the language context \p CTX to \p ND.
 *
 * \param ctx the language context that is to be modified.
 * \param nd the node that is to be made the current one. */
extern_inline void lang_om_set_node(lang_ctx_t ctx, xmlNodePtr nd);
/**
 * Return the value of the attribute \p ATTR of the current node of \p CTX.
 * If the current node of \p CTX has no attribute \p ATTR, \c NULL is
 * returned.
 *
 * \param ctx the language context.
 * \param attr the name of the attribute to be used as a key for the lookup.
 * \return a \c void* with the value of the attribute or \c NULL if none. */
extern_inline void *lang_om_get_node_attr(lang_ctx_t ctx, const char *attr);
/**
 * Return the current value of the scratch node set in the OM data of \p CTX.
 * This function is recommended as its macro version harbours the danger of
 * accidental modification or wrong casting.
 * 
 * \note This function does \e NOT check if the input arg actually comprises
 *   an openmath context.
 *
 * \param ctx the language context
 * \return the \c scratch_ns slot of the openmath context inside \p CTX */
extern_inline xmlNodeSetPtr lang_om_get_scratch_ns(lang_ctx_t ctx);


/**
 * An openmath object context. ..
 * comprising a look-ahead view on the tree to be parsed. */
struct om_ctx_s {
	/**
	 * Document to be parsed, consisting of all nodes. */
	xmlDocPtr curdoc;
	/**
	 * Current node to be parsed. */
	xmlNodePtr curnd;
	/**
	 * Scratch context for xpath expressions.
	 * this can be assumed to be initialised properly. */
	xmlXPathContextPtr scratch_xpc;
	/**
	 * Scratch pointer to the object yielded from evaluating the
	 * xpath expression. */
	xmlXPathObjectPtr scratch_xpo;
	/**
	 * Scratch pointer to a node set yielded from eval'ing the
	 * xpath expression. */
	xmlNodeSetPtr scratch_ns;
	/**
	 * blabla */
	xmlDocPtr scratch_doc;
	/**
	 * Generic data cell.
	 * the implementation may use this for whatever but
	 * there is no further protection of this cell. */
	void *data;
};

/**
 * Given a \p lang_data_t object, refer to the context specific to the
 * openmath language. */
#define om_ctx(_x)		((om_ctx_t)((_x)->data))
/**
 * Given a \c om_ctx_t object, refer to the current node. */
#define omctx_curnd(_x)		((_x)->curnd)
/**
 * Given a \c om_ctx_t object, refer to the scratch XPath context. */
#define omctx_scratch_xpc(_x)	((_x)->scratch_xpc)
/**
 * Given a \c om_ctx_t object, refer to the scratch XPath object. */
#define omctx_scratch_xpo(_x)	((_x)->scratch_xpo)
/**
 * Given a \c om_ctx_t object, refer to the scratch node set. */
#define omctx_scratch_ns(_x)	((_x)->scratch_ns)
/**
 * Given a \c om_ctx_t object, refer to the XML document. */
#define omctx_scratch_doc(_x)	((_x)->scratch_doc)

/* convenience */
/**
 * Given a \c lang_ctx_t with OM data, refer to the current node.
 * This is a convenience macro.  Its use is highly advised as it covers
 * the very common case of a language context whose private data slot
 * is an openmath context. */
#define lctx_om_curnd(_x)	(om_ctx(_x)->curnd)
/**
 * Given a \c lang_ctx_t with OM data, refer to the scratch XPath context.
 * This is a convenience macro.  Its use is highly advised as it covers
 * the very common case of a language context whose private data slot
 * is an openmath context. */
#define lctx_om_scratch_xpc(_x)	(om_ctx(_x)->scratch_xpc)
/**
 * Given a \c lang_ctx_t with OM data, refer to the scratch XPath object.
 * This is a convenience macro.  Its use is highly advised as it covers
 * the very common case of a language context whose private data slot
 * is an openmath context. */
#define lctx_om_scratch_xpo(_x)	(om_ctx(_x)->scratch_xpo)
/**
 * Given a \c lang_ctx_t with OM data, refer to the scratch node set.
 * This is a convenience macro.  Its use is highly advised as it covers
 * the very common case of a language context whose private data slot
 * is an openmath context. */
#define lctx_om_scratch_ns(_x)	(omctx_scratch_ns(om_ctx(_x)))
/**
 * Given a \c lang_ctx_t with OM data, refer to the XML document.
 * This is a convenience macro.  Its use is highly advised as it covers
 * the very common case of a language context whose private data slot
 * is an openmath context. */
#define lctx_om_scratch_doc(_x)	(om_ctx(_x)->scratch_doc)


/**
 * Register \p F as a dispatch function for objects of type \p TYPE. */
extern lnghdl_dispatch_f
defom_dispatch_f(long unsigned int type, lnghdl_dispatch_f f);
/**
 * Register \p F as a read function for tag \p TAG. */
extern lnghdl_read_f
defom_read_f(const char *tag, lnghdl_read_f f);

/**
 * Convert handle \p HDL into the language specific representation and
 * return its serialisation.
 * This is like lang_om_dispatch() but without initialisation steps.
 *
 * \param ctx the language context
 * \param hdl the libkant object to serialise */
extern void *lang_om_dispatch_internal(lang_ctx_t ctx, handle_t hdl);
/**
 * Convert handle \p HDL into the language specific representation and put
 * its serialisation into the \c curbuf slot of \p CTX.
 *
 * \param ctx the language context
 * \param hdl the libkant object to serialise */
extern void *lang_om_dispatch(lang_ctx_t ctx, handle_t hdl);

/**
 * Convert a buffer inside \p CTX into an internal handle and return it.
 *
 * \param ctx the language context
 * \return a libkant object or \c NULL if an error occurred */
extern handle_t lang_om_read(lang_ctx_t ctx);

/**
 * Uncloak, that is stringify, the private data inside \p CTX into \p BUF.
 * This is to gain insight into private structures.
 *
 * \param[out] buf a pointer to the buffer that is to be filled
 * \param[in] ctx the language context */
extern void lang_om_uncloak(char **buf, lang_ctx_t ctx);


/**
 * Initialise openmath language. */
extern void init_lang_om(void);
/**
 * Reinitialise openmath language. */
extern void reinit_lang_om(void);
/**
 * Deinitialise openmath language. */
extern void deinit_lang_om(void);

/** @} */


/* some openmaths syntax */
#define USE_XPATH	1
#define USE_LOCAL_XSD	1

#ifdef USE_LOCAL_XSD
#define OPENMATH1_XSD							\
	"openmath1.xsd"
#define OPENMATH2_XSD							\
	"openmath2.xsd"
#else  /* !USE_LOCAL_XSD */
#define OPENMATH1_XSD							\
	"http://www.openmath.org/standard/relaxng/openmath1.xsd"
#define OPENMATH2_XSD							\
	"http://www.openmath.org/standard/om20-2004-06-30/openmath2.xsd"
#endif	/* USE_LOCAL_XSD */

#define OM_HREF		"http://www.openmath.org/OpenMath"
#define OM_PREF		"OM"
#if 2				/* prefer om20 */
#define OM_XSD_LOC							\
	OM_HREF " "							\
	"http://www.openmath.org/standard/om20-2004-06-30/openmath2.xsd"
#elif 1
#define OM_XSD_LOC							\
	OM_HREF " "							\
	"http://www.openmath.org/standard/relaxng/openmath1.xsd"
#endif

#define ERR_HREF	"http://www.w3.org/2005/xqt-errors"
#define ERR_PREF	"err"

#define OMOBJ_TAG	OM_PREF ":OMOBJ"
#define OMATTR_TAG	OM_PREF ":OMATTR"
#define OMATP_TAG	OM_PREF ":OMATP"
#define OMA_TAG		OM_PREF ":OMA"
#define OME_TAG		OM_PREF ":OME"
#define OMS_TAG		OM_PREF ":OMS"
#define OMSTR_TAG	OM_PREF ":OMSTR"
#define OMI_TAG		OM_PREF ":OMI"
#define OMF_TAG		OM_PREF ":OMF"
#define OMBIND_TAG	OM_PREF ":OMBIND"
#define OMBVAR_TAG	OM_PREF ":OMBVAR"
#define OMV_TAG		OM_PREF ":OMV"
#define OMR_TAG		OM_PREF ":OMR"

#define PX_OMATP_PATH	"/OM:OMOBJ/OMATTR/OMATP"
#define PX_OMA_PATH	"/OM:OMOBJ/OMATTR/OMA"

#define expect_nodeset(_xop)				\
	if (LIKELY((_xop) &&				\
		   (_xop)->type == XPATH_NODESET &&	\
		   (_xop)->nodesetval != NULL));	\
	else

#define expect_nodeset_of_size(_xop, _cond)		\
	if (LIKELY((_xop) &&				\
		   (_xop)->type == XPATH_NODESET &&	\
		   (_xop)->nodesetval != NULL &&	\
		   (_cond)));				\
	else

#define expect_number(_xop)				\
	if (LIKELY(xop &&				\
		   xop->type == XPATH_NUMBER));		\
	else


/* head fiddling */
/**
 * Global table of heads.
 * A head is a pair (group, fun) where GROUP is the group blabla
 * and FUN is the name of a function therein. */
extern ase_htable_t heads;
/** convenience */
typedef struct kant_cd_s *kant_cd_t;
/** convenience */
typedef struct kant_cd_symbol_s *kant_cd_symbol_t;
/**
 * List of roles a symbol (in a head) can take. */
typedef enum role_e role_t;

/**
 * Generic type of callback functions inside heads.
 * Such a callback function takes any number of arguments and returns a
 * handle_t object.  The array of input argument is pointed to by the
 * first argument and its size goes into the second slot. */
typedef handle_t(*kant_head_funcall_f)(void*[], size_t);

/**
 * Return the CD indicated by \p NAME inside the global heads table.
 *
 * \param name the name of the CD
 * \return a kant_cd_t object if found, \c NULL otherwise
 **/
extern_inline kant_cd_t
kant_find_cd(const char *name);
/**
 * Return the symbol named \p SYMBOL in the cd named \p CD.
 *
 * \param cd the name of the CD
 * \param name the name of the symbol
 * \return a kant_cd_symbol_t object if found, \c NULL otherwise
 **/
extern_inline kant_cd_symbol_t
kant_find_head(const char *cd, const char *name);
/* them alternatives */
/**
 * Put the dictionary \p CD into the global heads table.
 *
 * \param cd the CD object to store
 * \return the input CD object if successful, or if a CD of the same name
 *   has been registered already, their CD object
 **/
extern kant_cd_t kant_register_cd(kant_cd_t cd);
/**
 * Put the symbol \p SYM into the global heads table.
 * \note This function overwrites any previously stored symbols of the
 *   same name.
 *
 * \param sym the head object (kant_cd_symbol_t) to store
 **/
extern void kant_register_symbol(kant_cd_symbol_t sym);

/**
 * List of roles a symbol (in a head) can take. */
enum role_e {
	ROLE_UNKOWN,
	ROLE_APPLICATION,
	ROLE_SYMBOL,
	ROLE_ATTR,
	NUMBER_OF_ROLES
};

/**
 * Object to store CDs for their use inside the heads table. */
struct kant_cd_s {
	/** Name of the CD. */
	/* const */ char *name;
	/** Description of the CD. */
	char *description;
	/** Whether or not this CD is transient.
	 * Transient means that there is no official ocd file
	 * describing the CD formally. */
	bool transientp;
	/** URL of the CD iff not transient. */
	const char *url;
	/** Base URI of the CD iff not transient. */
	const char *base;
};

/**
 * Object to store symbols for their use inside the heads table. */
struct kant_cd_symbol_s {
	/** Parent CD of this symbol.
	 * This imposes a tree-like hierarchy. */
	kant_cd_t cd;
	/** Name of the symbol. */
	/* const */ char *name;
	/** Description of the symbol. */
	char *description;
	/** Role of the symbol. */
	role_t role;
	/** Number of arguments that this symbol takes at least. */
	long int min_args;
	/** Number of arguments that this symbol takes at most.
	 * Use -1UL to indicate an infinite number of args. */
	long int max_args;		/* -1 for infinity */
	/** Associated call-back function. */
	handle_t(*fun)(lang_ctx_t);
};

/** @} */

extern_inline kant_cd_t
kant_find_cd(const char *name)
{
	size_t len = xstrlen(name);
	ase_hcode_t code = hcode_string(name, len);

	return ase_htable2_get(heads, code, 0UL);
}

extern_inline kant_cd_symbol_t
kant_find_head(const char *cd, const char *symbol)
{
	ase_hcode_t cd_code = hcode_string(cd, xstrlen(cd));
	ase_hcode_t sym_code = hcode_string(symbol, xstrlen(symbol));
	return ase_htable2_get(heads, cd_code, sym_code);
}

#define DEF_OMCD(_name, _description, _transientp, _url, _base)	\
	static struct kant_cd_s Qomcd_##_name = {		\
		.name = #_name,					\
		.description = _description,			\
		.transientp = _transientp,			\
		.url = _url,					\
		.base = _base,					\
	};							\
	static kant_cd_t Vomcd_##_name = NULL

#define DEF_OMREAD_F(_cd, _sym, min, max, _role, _desc, ctx)	\
	static handle_t						\
	Fom_read_##_cd##_##_sym(lang_ctx_t);			\
	static struct kant_cd_symbol_s Qomcd_##_cd##_##_sym = {	\
		.cd = &Qomcd_##_cd,				\
		.name = #_sym,					\
		.description = _desc,				\
		.role = _role,					\
		.min_args = min,				\
		.max_args = max,				\
		.fun = Fom_read_##_cd##_##_sym,			\
	};							\
	const kant_cd_symbol_t Vomcd_##_cd##_##_sym =		\
		&Qomcd_##_cd##_##_sym;				\
	static handle_t						\
	Fom_read_##_cd##_##_sym(lang_ctx_t ctx)

#define DEF_OMREAD_ALIAS_F(_cd, _sym, min, max, _r, _d, f)	\
	static struct kant_cd_symbol_s Qomcd_##_cd##_##_sym = {	\
		.cd = NULL,				\
		.name = #_sym,					\
		.description = _d,				\
		.role = _r,					\
		.min_args = min,				\
		.max_args = max,				\
		.fun = f,					\
	};							\
	const kant_cd_symbol_t Vomcd_##_cd##_##_sym =		\
		&Qomcd_##_cd##_##_sym

#define INIT_OMCD(_cd)					\
	Vomcd_##_cd = kant_register_cd(&Qomcd_##_cd)
#define INIT_OMREAD_F(_cd, _sym)					\
	do {								\
		Qomcd_##_cd##_##_sym.cd = Vomcd_##_cd;			\
		kant_register_symbol(Vomcd_##_cd##_##_sym);		\
	} while (0)


/* inlines */
/* actually we have to find us a sorta cool concept to cope with external
   dictionaries */
extern_inline void*
lang_om_make_symbol(void)
{
	return lang_xml_make_slot(OMS_TAG);
}

extern_inline void*
lang_om_symbol(const char *cd, const char *name)
{
	void *oms = lang_om_make_symbol();

	lang_xml_add_attr(oms, "cd", cd);
	lang_xml_add_attr(oms, "name", name);
	return oms;
}

extern_inline void*
lang_om_make_application(void)
{
	return lang_xml_make_slot(OMA_TAG);
}

extern_inline void*
lang_om_disp_symbol(void *oma, const char *cd, const char *name)
{
	void *child = lang_om_symbol(cd, name);
	lang_xml_add_child_slot(oma, child);
	return child;
}

extern_inline void*
lang_om_application(const char *cd, const char *name)
{
	void *oma = lang_om_make_application();
	lang_om_disp_symbol(oma, cd, name);
	return oma;
}

/* for bindings */
extern_inline void*
lang_om_make_binding(void)
{
	return lang_xml_make_slot(OMBIND_TAG);
}

/* huh? */
extern_inline void*
lang_om_binding(const char *cd, const char *name)
{
	void *ombind = lang_om_make_binding();
	lang_om_disp_symbol(ombind, cd, name);
	return ombind;
}

extern_inline void*
lang_om_make_string(void)
{
	return lang_xml_make_slot(OMSTR_TAG);
}

extern_inline void*
lang_om_string(const char *string, size_t len)
{
	void *omstr = lang_om_make_string();
	lang_xml_set_slot_data(omstr, string, len);
	return omstr;
}

extern_inline void*
lang_om_make_ombvar(void)
{
	return lang_xml_make_slot(OMBVAR_TAG);
}

extern_inline void*
lang_om_make_varsym(void)
{
	return lang_xml_make_slot(OMV_TAG);
}

extern_inline void*
lang_om_varsym(const char *varsym)
{
	void *omv = lang_om_make_varsym();
	lang_xml_add_attr(omv, "name", varsym);
	return omv;
}

extern_inline void*
lang_om_make_omi(void)
{
	return lang_xml_make_slot(OMI_TAG);
}

extern_inline void*
lang_om_omi(long int value)
{
	void *omi = lang_om_make_omi();
	char valstr[64];
	size_t vallen;

	vallen = snprintf(valstr, countof(valstr), "%ld", value);
	lang_xml_set_slot_data(omi, valstr, vallen);
	return omi;
}

extern_inline void*
lang_om_bind_to(const char *varsym)
{
	void *ombvar = lang_om_make_ombvar();
	void *omv = lang_om_varsym(varsym);
	lang_xml_add_child_slot(ombvar, omv);
	return ombvar;
}

/* for references */
extern_inline void*
lang_om_make_reference(const char *varsym)
{
	void *omr = lang_xml_make_slot(OMR_TAG);
	lang_xml_add_attr(omr, "xref", varsym);
	return omr;
}

/* error cd */
extern_inline void*
lang_om_make_error(void)
{
	return lang_xml_make_slot(OME_TAG);
}

extern_inline void*
lang_om_error_string(const char *errmsg, size_t len)
{
	void *oms = lang_om_symbol("moreerrors", "algorithm");
	void *omstr = lang_om_string(errmsg, len);

	/* nest the guys */
	lang_xml_add_sibling_slot(oms, omstr);
	return oms;
}

extern_inline void*
lang_om_error(const char *cd, const char *name)
{
	return lang_om_symbol(cd, name);
}

/* this one is NOUGHT in the error CD 
 * fixed as of 2007/Nov when Dan submitted scscp1
 */
extern_inline void*
lang_om_error_runtime(void)
{
/* Generate an OME cell over the error_runtime@scscp1 symbol
 * this cell in turn is sposed to be enriched with an OMSTR tag with a
 * verbose description of what's happened */
	return lang_om_symbol("scscp1", "error_runtime");
}

/* official error, see error.ocd */
extern_inline void*
lang_om_error_unhandled_symbol(void *sym)
{
/* Generate an OME cell over the unhandled_symbol@error symbol
 * if SYM is non-NULL it should be a node pointer to the symbol that
 * cannot be handled and hence triggered this error handler */
	void *oms = lang_om_symbol("error", "unhandled_symbol");
	if (LIKELY(sym != NULL)) {
		lang_xml_add_sibling_slot(oms, sym);
	}
	return oms;
}

/* official error, see error.ocd */
extern_inline void*
lang_om_error_unexpected_symbol(void *sym)
{
/* Generate an OME cell over the unexpected_symbol@error symbol
 * if SYM is non-NULL it should be a node pointer to the symbol that
 * was unexpected and hence triggered this error handler */
	void *oms = lang_om_symbol("error", "unexpected_symbol");
	if (LIKELY(sym != NULL)) {
		lang_xml_add_sibling_slot(oms, sym);
	}
	return oms;
}

/* official error, see error.ocd */
extern_inline void*
lang_om_error_unsupported_CD(void *sym)
{
/* Generate an OME cell over the unsupported_CD@error symbol
 * if SYM is non-NULL it should be a node pointer to the symbol
 * (as in OMS) whose cd attribute contains the unsupported dictionary
 * and hence triggered this error handler */
	void *oms = lang_om_symbol("error", "unsupported_CD");
	if (LIKELY(sym != NULL)) {
		lang_xml_add_sibling_slot(oms, sym);
	}
	return oms;
}

extern_inline void*
lang_om_attribeautify_node(const void *restrict*atp, void *node)
{
/* Return NODE wrapped in an OMATTR attribution.
 * As of OM2 we can attach attributes to every node along the path,
 * if optional ATP pointer-to-pointer is non-NULL it will be filled
 * with the pointer to the ATP group of the attribution */
	void *attr = lang_xml_make_slot(OMATTR_TAG);
	void *attr_pairs = lang_xml_make_slot(OMATP_TAG);

	lang_xml_add_child_slot(attr, attr_pairs);
	if (LIKELY(atp != NULL)) {
		*atp = attr_pairs;
	}
	if (LIKELY(node != NULL)) {
		lang_xml_add_child_slot(attr, node);
	}
	return attr;
}

/* accessors for protx ctx */
extern_inline void*
lang_om_get_node_content(lang_ctx_t ctx)
{
	return xmlNodeGetContent(lctx_om_curnd(ctx));
}

extern_inline void*
lang_om_get_node(lang_ctx_t ctx)
{
	return lctx_om_curnd(ctx);
}

extern_inline void
lang_om_set_node(lang_ctx_t ctx, xmlNodePtr nd)
{
	lctx_om_curnd(ctx) = nd;
	return;
}

extern_inline void*
lang_om_get_node_attr(lang_ctx_t ctx, const char *attr)
{
	return lxml_attr(lctx_om_curnd(ctx), attr);
}

extern_inline xmlNodeSetPtr
lang_om_get_scratch_ns(lang_ctx_t ctx)
{
	return lctx_om_scratch_ns(ctx);
}


/* scscp-tes crap */
/**
 * Register \p FUN as dispatcher function for the TES type \p TYPE.
 *
 * The distinction between kant types and TES types is still necessary.
 * Using the grand unified type system in libkant this distinction will
 * disappear and there will be only one big union of types.
 *
 * For the moment, this function is used to register a function that is
 * used as a callback whenever objects of the kant type TES happen to be
 * dispatched.  The subtype \p TYPE of the object will then be used to
 * decide which function to use to dispatch the object.
 *
 * \param type the TES type to register the dispatcher with
 * \param fun the function used as a callback
 * \return \c true if the registration was successful, \c false otherwise */
extern bool def_om_tes_disp_f(long int type, lnghdl_dispatch_f fun);
/**
 * Return the dispatcher function associated with TES subtype \p TYPE.
 *
 * This is the getter routine of def_om_tes_disp_f() and used internally.
 *
 * \param type the TES subtype
 * \return the dispatcher function as previously registered */
extern lnghdl_dispatch_f lang_om_tes_disp_f(long int type);

/**
 * \} */

#endif	/* INCLUDED_lang_om_h */
