/*** lang-xml.h -- XML support (needed for lang-protx)
 *
 * Copyright (C) 2006, 2007, 2008 Sebastian Freundt
 *
 * Author:  Sebastian Freundt <hroptatyr@sxemacs.org>
 *
 * This file is part of SXEmacs.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * 3. Neither the name of the author nor the names of any contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR "AS IS" AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
 * IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ***/

#ifndef INCLUDED_lang_xml_h
#define INCLUDED_lang_xml_h

#if defined(HAVE_LIBXML_TREE_H) && defined(HAVE_LIBXML_PARSER_H)
#include <libxml/tree.h>
#include <libxml/parser.h>
#include <libxml/xmlreader.h>
#include <libxml/xmlwriter.h>
#include <libxml/xmlschemas.h>
#else
#  error "How did you manage to reach this, moron?!"
#endif


#ifdef ALL_DEBUG_FLAGS
#undef XML_DEBUG_FLAG
#define XML_DEBUG_FLAG
#endif

/**
 * \defgroup lang_xml The XML Language
 * \ingroup langs
 *
 * \brief General XML stuff
 *
 * \{ */

/**
 * macro to output debugging information related to XML features */
#define LANG_DEBUG_XML(args...)		LANG_DEBUG("[xml] ")
/**
 * macro to output debugging information related to XML worker features */
#define LANG_DEBUG_XML_WORKER(args...)	LANG_DEBUG("[xml/worker]: " args)
/**
 * macro to output debugging information related to XML scratch features */
#define LANG_DEBUG_XML_SCRATCH(args...)	LANG_DEBUG("[xml/scratch]: " args)
/**
 * macro to output critical messages related to XML features */
#define LANG_CRITICAL_XML(args...)	LANG_CRITICAL("[xml] " args)

/**
 * URI to the XPath function schema */
#define FN_HREF		"http://www.w3.org/2005/xpath-functions"
/**
 * Name space prefix used for XPath functions */
#define FN_PREF		"fn"

/**
 * URI to the XML schema schema */
#define XS_HREF		"http://www.w3.org/2001/XMLSchema"
/**
 * Name space prefix used for XML schemas */
#define XS_PREF		"xs"

/**
 * URI to the XML schema instance schema */
#define XSI_HREF	"http://www.w3.org/2001/XMLSchema-instance"
/**
 * Name space prefix used for XML schema instances */
#define XSI_PREF	"xsi"
/**
 * buggered if I know */
#define XSI_SCHEMA_LOCATION_NAME		\
	"schemaLocation"

#define XML_PROP_STANDALONE			\
	"standalone"
#define XML_PROP_ENCODING			\
	"encoding"

#define XML_DEFAULT_VERSION		"1.0"
#define XML_DEFAULT_ENCODING		"ISO-8859-1"
#define XML_DEFAULT_STANDALONE_STATUS	"yes"

#define USE_XMLWRITER		1

typedef struct xml_pubctx_s *xml_pubctx_t;

/** \internal */
extern_inline void step_xml_bucket(lang_ctx_t ctx, size_t size);
/** \internal */
extern_inline void
xml_inject_tag_open(lang_ctx_t ctx, const char *tag, size_t taglen);
/** \internal */
extern_inline void
xml_inject_tag_close(lang_ctx_t ctx, const char *tag, size_t);
/** \internal */
extern_inline void
xml_inject_content(lang_ctx_t ctx,
		   const char *tag, size_t taglen,
		   const char *content, size_t contentlen);
/** \internal */
extern_inline void
xml_inject_fcontent(lang_ctx_t ctx,
		    const char *tag, size_t taglen,
		    lnghdl_dispatch_f f, handle_t hd);
/** \internal */
extern_inline void lang_xml_step_bucket(lang_ctx_t ctx);
/** \internal */
extern_inline void *lang_xml_save_excursion(lang_ctx_t ctx);
/** \internal */
extern_inline void lang_xml_reset_excursion(lang_ctx_t ctx, void *data);
/* writers */
/**
 * Return a newly created XMLDoc structure.
 *
 * \return an \c XMLDoc structure cast to a \c void* */
extern_inline void *lang_xml_make_doc(void);
/**
 * Given an empty XML document \p DOC, fill it with the node \p OBJ.
 *
 * \param doc a pointer to an XML document
 * \param obj a pointer to an XML node */
extern_inline void lang_xml_fill_doc_with_node(xmlDocPtr doc, xmlNodePtr obj);
/**
 * Combined action of lang_xml_make_doc() and lang_xml_fill_doc_with_node(). */
extern_inline void *lang_xml_make_doc_from_node(xmlNodePtr obj);
/**
 * Output the printed representation of \p DOC into \p BUFFER.
 *
 * \param buffer a pointer to a string whither the output goes
 * \param doc a pointer to an XML document that is to be dumped
 * \return the number of bytes filled into \p BUFFER */
extern_inline size_t lang_xml_dump_doc(char **buffer, xmlDocPtr doc);
/**
 * Like lang_xml_dump_doc() but with nice indentation and whatnot.
 *
 * \param buffer a pointer to a string whither the output goes
 * \param doc a pointer to an XML document that is to be dumped
 * \return the number of bytes filled into \p BUFFER */
extern_inline size_t lang_xml_pretty_dump_doc(char **buffer, xmlDocPtr doc);
/**
 * Return an XML element named \p NAME. */
extern_inline void *lang_xml_make_slot(const char *name);
/**
 * Set the contents of \p SLOT to a copy of \p D of size \p L. */
extern_inline void lang_xml_set_slot_data(void *slot, const char *d, size_t l);
/**
 * Add a copy of \p DATA (of size \p LEN) to the contents of \p SLOT. */
extern_inline void lang_xml_add_slot_data(void *slot, char *data, size_t len);
/**
 * Given a language context \p C and a libkant handle object \p H, this sets
 * the contents of \p S to the serialised representation of \p H. */
extern_inline void lang_xml_set_slot_hddata(lang_ctx_t c, void *s, handle_t h);
/**
 * Nest the XML element \p CHILD and \p SLOT.
 * \p CHILD becomes the right/bottom-most (latest) child of \p SLOT. */
extern_inline void lang_xml_add_child_slot(void *slot, void *child);
/**
 * Nest the XML element \p SIBLING and \p SLOT.
 * \p SIBLING becomes an immediate right neighbour of \p SLOT.
 * \note really? */
extern_inline void lang_xml_add_sibling_slot(void *slot, void *sibling);
/**
 * Given a language context \p CTX, add \p SLOT as child of the \c curnode
 * slot of \p CTX, or, if \p CTX has no nodes at all, add \p SLOT as root
 * element of the \c curdoc slot of \p CTX. */
extern_inline void lang_xml_put_slot(lang_ctx_t ctx, void *slot);
/**
 * Document me. */
extern_inline void
lang_xml_inject(lang_ctx_t, void*, lnghdl_dispatch_f, handle_t);
/**
 * Add an attribute with key \p K and value \p V to \p SLOT. */
extern_inline void *lang_xml_add_attr(void *slot, const char *k, const char *v);
/**
 * Add an attribute with key \p K and numeric value \p V to \p SLOT. */
extern_inline void *lang_xml_add_attr_L(void *slot, const char *k, long int v);
/**
 * Add a name space to \p SLOT.
 * The URI of the name space definition is specified by the \p REF argument,
 * the prefix to be used is specified by \p P. */
extern_inline void *lang_xml_add_ns(void *slot, const char *ref, const char *p);
/**
 * Add a schema definition to \p SLOT.
 * The URI of the schema definition is specified by the \p LOCATION argument. */
extern_inline void *lang_xml_add_schema_loc(void *slot, const char *location);
/* readers */
/**
 * Return the right neighbour of \p ND. */
extern_inline xmlNodePtr lxml_next_sibling(xmlNodePtr nd);
/**
 * Return the left neighbour of \p ND. */
extern_inline xmlNodePtr lxml_prev_sibling(xmlNodePtr nd);
/**
 * Return the earliest (top/left-most) child of of \p ND. */
extern_inline xmlNodePtr lxml_child(xmlNodePtr nd);
/**
 * Return the parent node of \p ND. */
extern_inline xmlNodePtr lxml_parent(xmlNodePtr nd);
/**
 * Return the content of \p ND. */
extern_inline char *lxml_content(xmlNodePtr nd);
/**
 * Return the value of the attribute key \p ATTR in the element \p ND. */
extern_inline char *lxml_attr(xmlNodePtr nd, const char *attr);
/**
 * Return the number of siblings of the XML element \p ND. */
extern_inline size_t lxml_count_siblings(xmlNodePtr nd);
/**
 * Return the right neighbour (successor) of the current node in \p CTX. */
extern_inline void *lang_xml_next_sibling(lang_ctx_t ctx);
/**
 * Return the left neighbour (predecessor) of the current node in \p CTX. */
extern_inline void *lang_xml_prev_sibling(lang_ctx_t ctx);
/**
 * Return the earliest (top/left-most) child of the current node in \p CTX. */
extern_inline void *lang_xml_child(lang_ctx_t ctx);
/**
 * Return the parent node of the current node in \p CTX. */
extern_inline void *lang_xml_parent(lang_ctx_t ctx);


struct xml_pubctx_s {
	void *curdoc;
	void *curnode;
};

/**
 * Register \p F as a dispatch function for objects of type \p TYPE. */
extern lnghdl_dispatch_f
defxml_dispatch_f(long unsigned int type, lnghdl_dispatch_f f);
/**
 * Register \p F as a read function for tag \p TAG. */
extern lnghdl_read_f
defxml_read_f(const char *tag, lnghdl_read_f f);

/**
 * Convert handle \p HDL into the language specific representation and
 * return its serialisation.
 * This is like lang_xml_dispatch() but without initialisation steps.
 *
 * \param ctx the language context
 * \param hdl the libkant object to serialise */
extern void *lang_xml_dispatch_internal(lang_ctx_t ctx, handle_t hdl);
/**
 * Convert handle \p HDL into the language specific representation and put
 * its serialisation into the \c curbuf slot of \p CTX.
 *
 * \param ctx the language context
 * \param hdl the libkant object to serialise */
extern void *lang_xml_dispatch(lang_ctx_t ctx, handle_t hdl);

/**
 * \internal
 * \deprecated */
extern handle_t lang_xml_read_slot(lang_ctx_t ctx, const char *slot);
/**
 * \internal
 * \deprecated */
extern handle_t lang_xml_read_slotchild(lang_ctx_t ctx, const char *slot);
/**
 * Convert a buffer inside \p CTX into an internal handle and return it.
 *
 * \param ctx the language context
 * \return a libkant object or \c NULL if an error occurred */
extern handle_t lang_xml_read(lang_ctx_t ctx);

/**
 * Uncloak, that is stringify, the private data inside \p CTX into \p BUF.
 * This is to gain insight into private structures.
 *
 * \param[out] buf a pointer to the buffer that is to be filled
 * \param[in] ctx the language context */
extern void lang_xml_uncloak(char **buf, lang_ctx_t ctx);


/**
 * Initialise xml language. */
extern void init_lang_xml(void);
/**
 * Reinitialise xml language. */
extern void reinit_lang_xml(void);
/**
 * Deinitialise xml language. */
extern void deinit_lang_xml(void);


/* inlines */
extern_inline void
step_xml_bucket(lang_ctx_t ctx, size_t size)
{
	ctx->offset += size;
	ctx->curbuf += size;
}

extern_inline void
xml_inject_tag_open(lang_ctx_t ctx, const char *tag, size_t taglen)
{
/* pump TAG of size TAGLEN into the XML stream */
	char *buf = ctx->curbuf;

	/* now push the opening tag */
	*buf = '<';
	xmemcpy(buf+1, tag, taglen);
	*(buf+taglen) = '>';
	step_xml_bucket(ctx, taglen+1);
	return;
}

extern_inline void
xml_inject_tag_close(lang_ctx_t ctx, const char *tag, size_t taglen)
{
/* pump TAG of size TAGLEN into the XML stream */
	char *buf = ctx->curbuf;

	/* now push the opening tag */
	*buf = '<';
	*(buf+1) = '/';
	xmemcpy(buf+2, tag, taglen);
	*(buf+taglen+1) = '>';
	step_xml_bucket(ctx, taglen+2);
	return;
}

extern_inline void
xml_inject_content(lang_ctx_t ctx,
		   const char *tag, size_t taglen,
		   const char *content, size_t contentlen)
{
	/* maybe resize? */
	resize_lang_ctx(ctx, contentlen+2*taglen+16);

	/* recompute the position of the current buffer */
	recomp_lang_ctx(ctx);
	/* now push the opening tag */
	xml_inject_tag_open(ctx, tag, taglen);
	/* copy the content */
	xmemcpy(ctx->curbuf, content, contentlen);
	step_xml_bucket(ctx, contentlen);
	/* finally the closing tag */
	xml_inject_tag_close(ctx, tag, taglen);
	return;
}

extern_inline void
xml_inject_fcontent(lang_ctx_t ctx,
		    const char *tag, size_t taglen,
		    lnghdl_dispatch_f f, handle_t hd)
{
	/* maybe resize? */
	resize_lang_ctx(ctx, 2*taglen+16);

	/* recompute the position of the current buffer */
	recomp_lang_ctx(ctx);
	/* now push the opening tag */
	xml_inject_tag_open(ctx, tag, taglen);
	/* copy the content */
	f(ctx, hd);
	/* finally the closing tag */
	xml_inject_tag_close(ctx, tag, taglen);
	return;
}


extern_inline void
lang_xml_step_bucket(lang_ctx_t ctx)
{
	xmlTextReaderPtr rd = ctx->data;
	xmlTextReaderRead(rd);
}

extern_inline void*
lang_xml_save_excursion(lang_ctx_t ctx)
{
	xml_pubctx_t ctxdata = ctx->data;
	return ctxdata->curnode;
}

extern_inline void
lang_xml_reset_excursion(lang_ctx_t ctx, void *data)
{
	xml_pubctx_t ctxdata = ctx->data;
	ctxdata->curnode = data;
	return;
}

/* writers */
extern_inline void*
lang_xml_make_doc(void)
{
	xmlDocPtr doc = xmlNewDoc((const xmlChar*)XML_DEFAULT_VERSION);

	/* fill in some sane values */
	/* brag about standalonelyness */
	doc->standalone = 1;
	return doc;
}

extern_inline void
lang_xml_fill_doc_with_node(xmlDocPtr doc, xmlNodePtr obj)
{
	xmlDocSetRootElement(doc, obj);
	return;
}

extern_inline void*
lang_xml_make_doc_from_node(xmlNodePtr obj)
{
	xmlDocPtr doc = lang_xml_make_doc();
	lang_xml_fill_doc_with_node(doc, obj);
	return doc;
}

extern_inline size_t
lang_xml_dump_doc(char **buffer, xmlDocPtr doc)
{
	int size = 0;
	xmlDocDumpFormatMemoryEnc(
		doc, (xmlChar**)buffer, &size,
		(const char*)XML_DEFAULT_ENCODING, 0);
	return (size_t)size;
}

extern_inline size_t
lang_xml_pretty_dump_doc(char **buffer, xmlDocPtr doc)
{
	int size = 0;
	xmlDocDumpFormatMemoryEnc(
		doc, (xmlChar**)buffer, &size,
		(const char*)XML_DEFAULT_ENCODING, 1);
	return (size_t)size;
}

extern_inline void*
lang_xml_make_slot(const char *name)
{
	return xmlNewNode(NULL, (const xmlChar*)name);
}

extern_inline void
lang_xml_set_slot_data(void *slot, const char *data, size_t len)
{
	xmlNodeSetContentLen(slot, (const xmlChar*)data, len);
	return;
}

extern_inline void
lang_xml_add_slot_data(void *slot, char *data, size_t len)
{
	xmlNodeAddContentLen(slot, (xmlChar*)data, len);
	return;
}

extern_inline void
lang_xml_set_slot_hddata(lang_ctx_t ctx, void *slot, handle_t hd)
{
	xml_pubctx_t ctxdata = ctx->data;
	void *excursion = ctxdata->curnode;

	ctxdata->curnode = slot;
	lang_xml_dispatch_internal(ctx, hd);
	ctxdata->curnode = excursion;
	return;
}

extern_inline void
lang_xml_add_child_slot(void *slot, void *child)
{
	xmlAddChildList(slot, child);
	return;
}

extern_inline void
lang_xml_add_sibling_slot(void *slot, void *sibling)
{
	xmlAddSibling(slot, sibling);
	return;
}

extern_inline void
lang_xml_put_slot(lang_ctx_t ctx, void *slot)
{
	xml_pubctx_t data = ctx->data;
	if (data->curnode == NULL) {
		data->curnode = xmlDocSetRootElement(data->curdoc, slot);
	} else {
		xmlAddChild(data->curnode, slot);
	}
}

extern_inline void
lang_xml_inject(lang_ctx_t ctx, void *slot,
		lnghdl_dispatch_f f, handle_t hd)
{
	xml_pubctx_t ctxdata = ctx->data;
	void *excursion = ctxdata->curnode;

	ctxdata->curnode = slot;
	f(ctx, hd);
	lang_xml_put_slot(ctx, slot);
	ctxdata->curnode = excursion;
	return;
}

extern_inline void*
lang_xml_add_attr(void *slot, const char *name, const char *value)
{
	return xmlNewProp(slot, (const xmlChar*)name, (const xmlChar*)value);
}

extern_inline void*
lang_xml_add_attr_L(void *slot, const char *name, long int value)
{
	char v[64];
	snprintf(v, countof(v), "%ld", value);
	return xmlNewProp(slot, (const xmlChar*)name, (const xmlChar*)v);
}

extern_inline void*
lang_xml_add_ns(void *slot, const char *href, const char *prefix)
{
	return xmlNewNs(slot, (const xmlChar*)href, (const xmlChar*)prefix);
}

extern_inline void*
lang_xml_add_schema_loc(void *slot, const char *location)
{
	void *xsi = lang_xml_add_ns(slot, XSI_HREF, XSI_PREF);
	return xmlNewNsProp(
		slot, xsi,
		(const xmlChar*)XSI_SCHEMA_LOCATION_NAME,
		(const xmlChar*)location);
}

/* readers */
extern_inline xmlNodePtr
lxml_next_sibling(xmlNodePtr nd)
{
	return nd->next;
}

extern_inline xmlNodePtr
lxml_prev_sibling(xmlNodePtr nd)
{
	return nd->prev;
}

extern_inline xmlNodePtr
lxml_child(xmlNodePtr nd)
{
	return nd->children;
}

extern_inline xmlNodePtr
lxml_parent(xmlNodePtr nd)
{
	return nd->parent;
}

extern_inline char*
lxml_content(xmlNodePtr nd)
{
	return (char*)xmlNodeGetContent(nd);
}

extern_inline char*
lxml_attr(xmlNodePtr nd, const char *attr)
{
	return (char*)xmlGetProp(nd, (const xmlChar*)attr);
}

extern_inline size_t
lxml_count_siblings(xmlNodePtr nd)
{
	size_t nargs = 1;

	if (nd == NULL) {
		return 0;
	}
	while ((nd = lxml_next_sibling(nd))) {
		nargs++;
	}
	return nargs;
}

extern_inline void*
lang_xml_next_sibling(lang_ctx_t ctx)
{
	xmlNodePtr nd = ctx->data;
	if (nd) {
		ctx->data = lxml_next_sibling(nd);
	}
	return nd;
}

extern_inline void*
lang_xml_prev_sibling(lang_ctx_t ctx)
{
	xmlNodePtr nd = ctx->data;
	if (nd) {
		ctx->data = lxml_prev_sibling(nd);
	}
	return nd;
}

extern_inline void*
lang_xml_child(lang_ctx_t ctx)
{
	xmlNodePtr nd = ctx->data;
	if (nd) {
		ctx->data = lxml_child(nd);
	}
	return nd;
}

extern_inline void*
lang_xml_parent(lang_ctx_t ctx)
{
	xmlNodePtr nd = ctx->data;
	if (nd) {
		ctx->data = lxml_parent(nd);
	}
	return nd;
}

static inline xmlNodePtr
lang_xml_skip_text_elems(xmlNodePtr nd)
{
	while (nd &&
	       (nd->type == XML_TEXT_NODE || nd->type == XML_COMMENT_NODE)) {
		nd = lxml_next_sibling(nd);
	}
	return nd;
}

/**
 * \} */

#endif	/* INCLUDED_lang_xml_h */
