/*** langs.h -- Various KASH languages
 *
 * Copyright (C) 2006, 2007, 2008 Sebastian Freundt
 *
 * Author:  Sebastian Freundt <hroptatyr@sxemacs.org>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * 3. Neither the name of the author nor the names of any contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR "AS IS" AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
 * IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ***/

#ifndef INCLUDED_langs_h
#define INCLUDED_langs_h

#include <kant/kant.h>
#include "events.h"
#include "protos.h"
#if defined HAVE_POPT && defined WITH_POPT
# include <popt.h>
#endif	/* HAVE_POPT && WITH_POPT */

#ifdef ALL_DEBUG_FLAGS
#undef LANG_DEBUG_FLAG
#define LANG_DEBUG_FLAG			1
#endif

/**
 * \defgroup langs AutoKASH Language support
 * \ingroup autokash
 *
 * \brief Defines common functionality for autokash languages.
 *
 * \{
 */

/** internal */
#define __LANG_DEBUG_CONT__(args...)	fprintf(stderr, args)
/** internal */
#define __LANG_DEBUG__(args...)		__LANG_DEBUG_CONT__("LANG " args)
#ifdef LANG_DEBUG_FLAG
/**
 * macro to output debugging information inside the language subsystem */
#define LANG_DEBUG(args...)		__LANG_DEBUG__(args)
/** internal */
#define LANG_DEBUG_CONT(args...)	__LANG_DEBUG_CONT__(args)
#else
/**
 * macro to output debugging information inside the language subsystem */
#define LANG_DEBUG(args...)
/** internal */
#define LANG_DEBUG_CONT(args...)
#endif
/**
 * macro to output debugging info for events inside the language subsystem */
#define LANG_DEBUG_EVENT(args...)	LANG_DEBUG("[event] " args)
/**
 * macro to output critical messages inside the language subsystem */
#define LANG_CRITICAL(args...)		__LANG_DEBUG__("CRITICAL: " args)

/**
 * pointer to the event handler for parse events */
extern event_handler_t Vevhdl_parse;
/**
 * pointer to the event handler for resultify events */
extern event_handler_t Vevhdl_resultify;
/**
 * pointer to the event handler for egest events */
extern event_handler_t Vevhdl_egest;

/** convenience */
typedef enum langs_e lang_t;
/** convenience */
typedef struct lang_handler_s *lang_handler_t;
/** convenience */
typedef struct lang_stream_s *lang_stream_t;
/** convenience */
typedef struct lang_ctx_s *lang_ctx_t;
/** convenience */
typedef struct lang_data_s *lang_data_t;

/**
 * Prototype for reader functions. */
typedef handle_t(*lnghdl_read_f)(lang_ctx_t);
/**
 * Prototype for dispatcher functions. */
typedef void*(*lnghdl_dispatch_f)(lang_ctx_t, handle_t);
/**
 * Prototype for parser functions. */
typedef int(*lnghdl_parse_f)(event_t);
/**
 * Prototype for resultifying functions. */
typedef int(*lnghdl_resultify_f)(lang_ctx_t, handle_t);
/**
 * Prototype for vanishing functions. */
typedef int(*lnghdl_vanish_f)(event_t);
/**
 * Prototype for freer functions. */
typedef void(*lnghdl_free_f)(event_t);
/**
 * Prototype for printer functions. */
typedef void(*lnghdl_prnt_f)(event_t);


/**
 * Enlarge the buffer in \p CTX by \p ADDSIZE
 *
 * \param ctx the language context
 * \param addsize the number of bytes to add */
extern_inline void resize_lang_ctx(lang_ctx_t ctx, size_t addsize);
/**
 * Realign the buffer of \p CTX, advancing it by the current offset. */
extern_inline void recomp_lang_ctx(lang_ctx_t ctx);
/* error handler */
/**
 * Inject a new error object created from \p ERRCELL into the language
 * context \p CTX.  The error is treated as level \p LEV error.
 *
 * \param ctx the language context
 * \param lev an error level
 * \param errcell arbitrary data */
extern_inline void
lang_error(lang_ctx_t ctx, kant_errlev_t lev, void *errcell);
/**
 * Inject a new error object created from an error message string \p MSG
 * of length \p LEN into the language context \p CTX.  The error is treated
 * as level \p LEV error.
 *
 * \param ctx the language context
 * \param lev an error level
 * \param msg a string with the error message
 * \param len the length of the string */
extern_inline void
lang_error_msg(lang_ctx_t ctx, kant_errlev_t lev, char *const msg, size_t len);


/**
 * Global symbols of supported languages.
 */
enum langs_e {
	lng_unknown,
	lng_kant,
	lng_kash,
	lng_xml,
	lng_om,
	NUMBER_OF_LANGS
};

/**
 * Compound for a language stream, just a type+size+data thing.
 */
struct lang_stream_s {
	/** the type of the language stream */
	long unsigned int type;
	/** the size of the language stream */
	long unsigned int size;
	/** arbitrary data */
	void *data;
};

/**
 * Compound for a language context.
 */
struct lang_ctx_s {
	/** the dispatch buffer */
	void *buffer;
	/** the overall size of the dispatch buffer */
	size_t size;
	/** the language */
	lang_t lang;
	/** state info, pointer to current buffer */
	void *curbuf;
	/** state info, offset in the current buffer */
	size_t offset;
	/** custom data */
	void *data;
	/** protocol specific data */
	void *proto_data;
	/** error data */
	kant_errchn_t errors;
};

/**
 * Given a \c lang_ctx_t object, refer to the language slot. */
#define lang_ctx_lang(_x)	((_x)->lang)
/**
 * Given a \c lang_ctx_t object, refer to the flags slot. */
#define lang_ctx_flags(_x)	((_x)->flags)


/** bollocks */
#define SIZE_LANG_STREAM_S	sizeof(struct lang_stream_s)
/** even more bollocks */
#define LANG_STREAM_S_FROBS	16*SIZE_LANG_STREAM_S

/**
 * Define a dispatch function.
 *
 * \param _name the name of the read function
 * \param _ctx the name of the language context variable
 * \param _hd the name of the object handle variable */
#define DEFDISPATCH_F(_name, _ctx, _hd)					\
	static void*							\
	_name(lang_ctx_t _ctx, handle_t _hd)

/**
 * Define an unused dispatch function.
 *
 * \param _name the name of the read function
 * \param _ctx the name of the language context variable
 * \param _hd the name of the object handle variable */
#define DEF_UNUSED_DISPATCH_F(_name, _ctx, _hd)				\
	static void*							\
	_name(lang_ctx_t _ctx, handle_t _hd)				\
		__attribute__((unused));				\
	static void*							\
	_name(lang_ctx_t _ctx, handle_t _hd)

/**
 * Define a read function.
 *
 * \param _name the name of the read function
 * \param _ctx the name of the language context variable */
#define DEFREAD_F(_name, _ctx)						\
	static handle_t							\
	_name(lang_ctx_t _ctx)


/**
 * Return a newly created context for language \p LNG.
 *
 * \param lng the language
 * \return a context for the specified language */
extern lang_ctx_t make_lang_ctx(lang_t lng);
/**
 * Free the resources in \p CTX.
 *
 * \param ctx the language context that is to be freed */
extern void free_lang_ctx(lang_ctx_t ctx);
/**
 * Register \p F as function to be used for serialisation of objects.
 *
 * \param lang the language to serialise into
 * \param type the type of the object that \p F serialises
 * \param f the function that does the serialisation
 * \return on success the function \p F is returned, \c NULL otherwise */
extern lnghdl_dispatch_f
deflang_dispatch_f(lang_t lang, long unsigned int type, lnghdl_dispatch_f f);
/**
 * Register \p F as function to be used to read objects.
 *
 * \param lang the language to deserialise from
 * \param data language specific identification of object types
 * \param f the function that does the deserialisation
 * \return on success the function \p F is returned, \c NULL otherwise */
extern lnghdl_read_f
deflang_read_f(lang_t lang, void *data, lnghdl_read_f f);

/**
 * Return a newly created parse event.
 *
 * Parsing is what happens after input is successfully read off a socket and
 * which complies with a given protocol.  The protocol must have determined the
 * language to be used by now and its data is passed as first argument, \p PD,
 * the chosen language is specified as second argument, \p LNG.
 *
 * \param pd the data of the protocol
 * \param lng the language to be used
 * \return an event ready to be queued */
extern event_t make_parse_event(const proto_data_t pd, lang_t lng);
/**
 * Return a newly created resultify event.
 *
 * Resultification is what happens after evaluation and is in fact the beginning
 * of the output chain.  A resultify event carries a new language context that
 * can be used for serialisation of the result.
 *
 * The new event is created from two previous ones, the eval event \p EVEV and
 * the parse event \p PRSEV.  The former is obviously needed because it holds
 * the actual result of a computation or the like.  The latter is needed because
 * the parse event holds information about the originally used protocol which in
 * turn keeps information about the input source.
 * As long as no advanced rerouting features are in effect the output can simply
 * be delivered to the asking client.  Regarding those rerouting features may
 * amend the meaning of the second argument and all the info necessary for the
 * output serialisation goes there.
 *
 * \param evev the event that was queued by an evaluation
 * \param prsev the event that caused parsing
 * \return an event ready to be queued */
extern event_t
make_resultify_event(const restrict event_t evev, const restrict event_t prsev);
/**
 * Convert handle \p HDL into the language specific representation and put
 * its serialisation into the \c curbuf slot of \p CTX.
 *
 * \param ctx the language context
 * \param hdl the libkant object to serialise */
extern lang_stream_t *lang_dispatch(lang_ctx_t ctx, handle_t hdl);
/**
 * Return a newly created egest event.
 *
 * Egesting is what happens after the resultification.  The evaluated object is
 * serialised into the language specified in the resultification event, \p REV.
 *
 * \param rev the event that was queued by a resultification
 * \return an event ready to be queued */
extern event_t make_egest_event(const restrict event_t rev);

/**
 * Store \p F as reader function (deserialisation) for language \p LNG */
extern void register_read_fun(lang_t lng, lnghdl_read_f f);
/**
 * Store \p F as dispatcher function (serialisation) for language \p LNG */
extern void register_dispatch_fun(lang_t lng, lnghdl_dispatch_f f);

#if defined HAVE_POPT && defined WITH_POPT
extern struct poptOption lang_opts[];
#endif
/**
 * Global variable to store the state of the -z option.
 * If set this will trigger the gap kludge to wipe name spaces. */
extern bool without_namespaces_p;

/**
 * Initialise the language subsystem. */
extern void init_langs(void);
/**
 * Reinitialise the language subsystem. */
extern void reinit_langs(void);
/**
 * Deinitialise the language subsystem. */
extern void deinit_langs(void);


/* more inlines */
extern_inline void
resize_lang_ctx(lang_ctx_t ctx, size_t addsize)
{
	if (ctx->size < ctx->offset + addsize) {
		/* realloc because two more handles follow */
		ctx->size += addsize + LANG_STREAM_S_FROBS;
		ctx->buffer = xrealloc(ctx->buffer, ctx->size);
	}
}

extern_inline void
recomp_lang_ctx(lang_ctx_t ctx)
{
	ctx->curbuf = (lang_stream_t*)(((char*)ctx->buffer) + ctx->offset);
}

/* error handler */
extern_inline void
lang_error(lang_ctx_t ctx, kant_errlev_t level, void *errcell)
{
	kant_error_t ec = make_kant_error_data(level, errcell, NULL);

	kant_errchn_push(ctx->errors, ec);
	return;
}

extern_inline void
lang_error_msg(lang_ctx_t ctx, kant_errlev_t level, char *const m, size_t l)
{
	kant_error_t ec = make_kant_error_msg(level, m, l);

	kant_errchn_push(ctx->errors, ec);
	return;
}

/**
 * \} */


#ifdef WITH_LANG_XML
#include "lang-xml.h"
#endif
#ifdef WITH_LANG_OM
#include "lang-om.h"
#include "protos.h"
#endif
#ifdef WITH_LANG_KANT
#include "lang-kant.h"
#endif


#endif	/* INCLUDED_langs_h */
