/*** modules.h -- the module loader
 *
 * Copyright (C) 2005, 2006, 2007, 2008 Sebastian Freundt
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * 3. Neither the name of the author nor the names of any contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR "AS IS" AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
 * IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * This file is part of autokash.
 *
 ***/

#ifndef INCLUDED_modules_h
#define INCLUDED_modules_h

#ifdef HAVE_EVENTS
# include "worker-asyneq.h"
#endif
#include "inputs.h"
#include "langs.h"
#include "protos.h"
#include <ltdl.h>

#ifdef ALL_DEBUG_FLAGS
#undef MODULE_DEBUG_FLAG
#define MODULE_DEBUG_FLAG		1
#endif

/**
 * \defgroup dso AutoKASH pluggable modules support
 * \ingroup autokash
 *
 * \brief Provides a plug-in like mechanism to add additional facilities.
 *
 * \{
 */

/** internal */
#define __MODULE_DEBUG__(args...)	fprintf(stderr, "MODULE " args)
#ifdef MODULE_DEBUG_FLAG
/**
 * macro to output debugging information inside the module subsystem */
#define MODULE_DEBUG(args...)		__MODULE_DEBUG__(args)
#else
/**
 * macro to output debugging information inside the module subsystem */
#define MODULE_DEBUG(args...)
#endif
/**
 * macro to output critical information inside the module subsystem */
#define MODULE_CRITICAL(args...)	__MODULE_DEBUG__("CRITICAL: " args)

/**
 * pointer to the event handler for load events
 *
 * load events may arise from certain input sequences, like scscp's load
 * instruction (<?scscp load ... ?>).  To be loaded modules on the command
 * line however do \e NOT trigger load events. */
extern event_handler_t Vevhdl_load;

/**
 * Open and load the module \p NAME.
 *
 * \note AutoKASH does not (yet) do shell globbing, nor does it search
 *   for modules along a module directory.
 *
 * \todo allow basic shell globbing
 * \todo search for modules in a module directory
 *
 * \param name the file name of the module to load
 * \return \c true if the module could be opened and loaded successfully,
 *   \c false otherwise */
extern bool open_aux(const char *name);
/**
 * Return a newly created load event.
 *
 * Load events can be used to load or unload modules during runtime.
 *
 * \todo conceive of ways to trigger module unloading too
 *
 * \param p the protocol
 * \param file the file name of the module to be loaded/unloaded
 * \return an event ready to be queued */
extern event_t make_load_event(proto_t p, const char *file);

/**
 * Helper macro for module authors.
 * This defines a simple state closure so that modules can check if they
 * have been loaded or not.
 *
 * \param _mod the module name as C symbol */
#define DEFMODULE_SUGAR(_mod)				\
	const char *_mod##_LTX_dependencies[] = {NULL};	\
							\
	static struct kmod_state_s {			\
			bool initialised;		\
			lt_dlhandle handle;		\
	} _##_mod##_state = {				\
		.initialised = false,			\
		.handle = NULL,				\
	}

/**
 * Helper macro for module authors.
 * This expands to a checking routine, if the module has not been loaded
 * yet, do so and keep track of this.  If the module has been loaded already
 * trigger a message.
 *
 * \todo pass status message back to the caller ... done
 * 
 * \param _mod the name of the module to supervise, a C symbol
 * \param args stuff to be done when loading the module for the first time */
#define INIT_MODULE_SUGAR(_mod, args...)		\
	if (_##_mod##_state.initialised == false) {	\
		args;					\
		_##_mod##_state.initialised = true;	\
		_##_mod##_state.handle = handle;	\
	} else {					\
		LANG_DEBUG_OM("module " #_mod " loaded already\n");	\
	}


/* initialisers */
/**
 * Initialise the language subsystem. */
extern void init_modules(void);
/**
 * Reinitialise the language subsystem. */
extern void reinit_modules(void);
/**
 * Deinitialise the language subsystem. */
extern void deinit_modules(void);

/**
 * \} */

#endif	/* INCLUDED_modules_h */
