/*** proto-scscp.h -- SCSCP on top of TCP/UDP {uni,any,multi}cast
 * Copyright (C) 2006, 2007, 2008 Sebastian Freundt
 *
 * Author:  Sebastian Freundt <hroptatyr@sxemacs.org>
 *
 * This file is part of SXEmacs.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * 3. Neither the name of the author nor the names of any contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR "AS IS" AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
 * IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ** Commentary:
 * This module deals with SCSCP processing instructions stuff on top of a
 * TCP/UDP {any,uni,multi}cast connection
 *
 * Two things here, really three
 * 1. session negotiation
 * 2. transaction blocks
 * 3. junk
 *
 ***/

#ifndef INCLUDED_proto_scscp_h
#define INCLUDED_proto_scscp_h

#include "config.h"
#include "events.h"
#include "protos.h"
#include "inputs.h"

#define PROTO_DEBUG_SCSCP(args...)	PROTO_DEBUG("[scscp] " args)
#define PROTO_DEBUG_SCSCP_130(args...)	PROTO_DEBUG("[scscp 1.3] " args)
#define PROTO_CRITICAL_SCSCP(args...)	PROTO_CRITICAL("[scscp] " args)

typedef struct proto_scscp_s *proto_scscp_t;
typedef struct proto_handler_s *proto_handler_t;

typedef enum scscp_version_e scscp_version_t;
typedef enum scscp_state_e scscp_state_t;

extern proto_t make_proto_scscp(void*);
extern void free_proto_scscp(proto_t);

extern event_t make_digest_event(proto_t);

extern void init_proto_scscp(void);
extern void deinit_proto_scscp(void);
extern void reinit_proto_scscp(void);

extern_inline void
proto_scscp_set_state(proto_scscp_t p, scscp_state_t state);
extern_inline void
proto_scscp_set_state_lock(proto_scscp_t p, scscp_state_t state);


enum scscp_version_e {
	SCSCP_VERSION_UNKNOWN,
	SCSCP_VERSION_110,
	SCSCP_VERSION_120,
	SCSCP_VERSION_130,
	SCSCP_VERSION_B0RKED,
	NUMBER_OF_SCSCP_VERSIONS
};

enum scscp_state_e {
	SCSCP_STATE_UNKNOWN,
	SCSCP_STATE_EHLO,
	SCSCP_STATE_NEGO,
	SCSCP_STATE_JUNK,
	SCSCP_STATE_DATA,	/* for OM data */
	SCSCP_STATE_QUIT,
	SCSCP_STATE_MORE,
	NUMBER_OF_SCSCP_STATES
};

struct proto_scscp_s {
	struct proto_s proto;
	scscp_version_t version;
	scscp_state_t state;
	char *message;
};

/* simple EHLO is
 * C--connects-->S
 * S----brags--->C: EHLO
 *
 * Official docs say:
 * Directly after the connection is established, the serversends its
   versionstring:

   Server: <?scscp system="{systemname}" system_version="{systemversion}"
   pid="{pid}" ?>

   where

   * {systemname} is a string containing the name of the CAS, e.g. "KANT",
     "GAP", "MuPAD", ...
   * {systemversion} is a string containig the version-number of the CAS,
     e.g. "11", "3.1", "4.1.5beta", ...
   * {pid} is an integer containing an identifier for the CAS ????

 *
 */

#define MAX_PI_SIZE		4096

#define PI_OPEN			"<?"
#define PI_CLOSE		"?>"
#define PI_SCSCP		"scscp"
#define PIATT_CAS_NAME		"service_name"
#define PIATT_CAS_VERSION	"service_version"
#define PIATT_CAS_ID		"service_id"
#define PIATT_SCSCP_VERSIONS	"scscp_versions"
#define PIATT_QUIT		"quit"
#define PIATT_CANCEL		"cancel"
#define PIATT_VERSION		"version"
#define PIATT_SECRET		"secret"
#define PIATT_REASON		"reason"
#define NEWLINE			"\n"
#define SPACE			" "

#define QUIT_MSG				\
	PI_OPEN					\
	PI_SCSCP " "				\
	PIATT_QUIT " "				\
	PI_CLOSE				\

#define CANCEL_MSG				\
	PI_OPEN					\
	PI_SCSCP " "				\
	PIATT_CANCEL " "			\
	PI_CLOSE				\

#define SCSCP_QUIT				\
	QUIT_MSG NEWLINE
#define SCSCP_CANCEL				\
	CANCEL_MSG NEWLINE

/* inlines */
extern_inline void
proto_scscp_set_state(proto_scscp_t p, scscp_state_t state)
{
/* assumes p's mutex is locked */
	p->state = state;
	return;
}

extern_inline void
proto_scscp_set_state_lock(proto_scscp_t p, scscp_state_t state)
{
	SXE_MUTEX_LOCK(&p->proto.mtx);
	proto_scscp_set_state(p, state);
	SXE_MUTEX_UNLOCK(&p->proto.mtx);
	return;
}

#endif	/* INCLUDED_proto_scscp_h */

/* proto-scscp.c ends here */
