/*** protos.h -- custom protocols on top of TCP/UDP {uni,any,multi}cast
 *
 * Copyright (C) 2006, 2007, 2008 Sebastian Freundt
 *
 * Author:  Sebastian Freundt <hroptatyr@sxemacs.org>
 *
 * This file is part of SXEmacs.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * 3. Neither the name of the author nor the names of any contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR "AS IS" AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
 * IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ***/

#ifndef INCLUDED_protos_h
#define INCLUDED_protos_h

#include "config.h"
#include "events.h"
#include <gc/cord.h>
#include "kash-system.h"

#ifdef ALL_DEBUG_FLAGS
#undef PROTO_DEBUG_FLAG
#define PROTO_DEBUG_FLAG		1
#endif

#define __PROTO_DEBUG__(args...)	fprintf(stderr, "PROTO " args)
#ifdef PROTO_DEBUG_FLAG
#define PROTO_DEBUG(args...)		__PROTO_DEBUG__(args)
#else
#define PROTO_DEBUG(args...)
#endif
#define PROTO_CRITICAL(args...)		__PROTO_DEBUG__("CRITICAL: " args)

typedef struct proto_s *proto_t;
typedef struct proto_data_s *proto_data_t;
typedef const struct proto_funs_s *proto_funs_t;
typedef enum kant_errlev_e proto_errlev_t;

typedef void(*proto_ehlo_f)(proto_t);
typedef void(*proto_quit_f)(proto_t, const char *reason);
typedef void(*proto_digest_f)(proto_t);
typedef void(*proto_egest_f)(proto_data_t);
typedef proto_data_t(*proto_make_data_f)(proto_t, size_t);
typedef void(*proto_free_data_f)(proto_data_t);
typedef void(*proto_error_f)(
	proto_data_t, proto_errlev_t, const char*, size_t);
typedef void(*proto_print_error_f)(
	proto_t, proto_errlev_t, const char*, size_t);

extern_inline size_t proto_buf_len(proto_t p);
extern_inline size_t proto_buf_len_lock(proto_t p);
extern_inline size_t proto_buf_len_cache(proto_t p);
extern_inline size_t proto_buf_len_cache_lock(proto_t p);
extern_inline void proto_buf_trunc(proto_t p);
extern_inline void proto_buf_trunc_lock(proto_t p);
extern_inline void proto_buf_cut(proto_t p, size_t start);
extern_inline void proto_buf_cut_lock(proto_t p, size_t start);
extern_inline void proto_buf_append_cord(proto_t p, CORD app);
extern_inline void proto_buf_append_string(proto_t p, const char *app);
extern_inline void proto_buf_prepend_cord(proto_t p, CORD prep);
extern_inline void proto_buf_prepend_string(proto_t p, const char *prep);
extern_inline void proto_buf_append_cord_lock(proto_t p, CORD app);
extern_inline void proto_buf_append_string_lock(proto_t p, const char *app);
extern_inline void proto_buf_prepend_cord_lock(proto_t p, CORD prep);
extern_inline void proto_buf_prepend_string_lock(proto_t p, const char *prep);


struct proto_funs_s {
	proto_ehlo_f ehlof;
	proto_quit_f quitf;
	proto_digest_f digestf;
	proto_egest_f egestf;
	proto_make_data_f make_data_f;
	proto_free_data_f free_data_f;
	/* to add an error to the protocol stack */
	proto_error_f errorf;
	/* to print an error immediately */
	proto_print_error_f print_error_f;
};

struct proto_s {
	int proto;
	proto_funs_t funs;

	size_t buf_len;
	CORD buf;

	/* move to data cell? */
	void *input_ctx;		/* input context */

	/* mutex to protect the proto against flood */
	sxe_mutex_t mtx;
};

#define proto_proto(_p)		((_p)->proto)
#define proto_functions(_p)	((_p)->funs)
#define proto_buffer_size(_p)	((_p)->buf_len)
#define proto_buffer(_p)	((_p)->buf)
#define proto_mutex(_p)		(&(_p)->mtx)
#define proto_input_ctx(_p)	((_p)->input_ctx)

struct proto_data_s {
	void *proto;
	sxe_mutex_t mtx;

	/* the language (a per-data thing) */
	void *lang;
	/* buffer */
	char *buffer;
	size_t size;
	/* errors */
	kant_errchn_t errors;
};

extern event_handler_t Vevhdl_proto;

/* handler funs */
typedef void(*prhdl_digest_f)(event_t);
typedef void(*prhdl_egest_f)(event_t);

struct proto_handler_s {
	prhdl_digest_f digest_f;
	prhdl_egest_f egest_f;
};

extern void prnt_proto_data(proto_data_t);

extern void init_protos(void);
extern void deinit_protos(void);
extern void reinit_protos(void);


/* inlines */
extern_inline size_t
proto_buf_len(proto_t p)
{
/* assumes p's mutex is locked */
	if (p->buf_len != (size_t)-1) {
		return p->buf_len;
	} else {
		return CORD_len(p->buf);
	}
}

extern_inline size_t
proto_buf_len_lock(proto_t p)
{
	size_t res;
	SXE_MUTEX_LOCK(&p->mtx);
	res = proto_buf_len(p);
	SXE_MUTEX_UNLOCK(&p->mtx);
	return res;
}

extern_inline size_t
proto_buf_len_cache(proto_t p)
{
/* assumes p's mutex is locked */
	if (p->buf_len != (size_t)-1) {
		return p->buf_len;
	} else {
		return p->buf_len = CORD_len(p->buf);
	}
}

extern_inline size_t
proto_buf_len_cache_lock(proto_t p)
{
	size_t res;
	SXE_MUTEX_LOCK(&p->mtx);
	res = proto_buf_len_cache(p);
	SXE_MUTEX_UNLOCK(&p->mtx);
	return res;
}

extern_inline void
proto_buf_trunc(proto_t p)
{
/* assumes p's mutex is locked */
	p->buf = CORD_EMPTY;
	p->buf_len = 0;
}

extern_inline void
proto_buf_trunc_lock(proto_t p)
{
	SXE_MUTEX_LOCK(&p->mtx);
	proto_buf_trunc(p);
	SXE_MUTEX_UNLOCK(&p->mtx);
	return;
}

extern_inline void
proto_buf_cut(proto_t p, size_t start)
{
/* assumes p's mutex is locked */
	(void)proto_buf_len_cache(p);
	p->buf = CORD_balance(CORD_substr(p->buf, start, p->buf_len));
	p->buf_len -= start;
	return;
}

extern_inline void
proto_buf_cut_lock(proto_t p, size_t start)
{
	SXE_MUTEX_LOCK(&p->mtx);
	proto_buf_cut(p, start);
	SXE_MUTEX_UNLOCK(&p->mtx);
	return;
}

extern_inline void
proto_buf_append_cord(proto_t p, CORD app)
{
/* assumes p's mutex is locked */
	p->buf = CORD_cat(p->buf, app);
	p->buf_len = (size_t)-1;
}

extern_inline void
proto_buf_append_string(proto_t p, const char *app)
{
	proto_buf_append_cord(p, CORD_from_char_star(app));
	return;
}

extern_inline void
proto_buf_prepend_cord(proto_t p, CORD prep)
{
/* assumes p's mutex is locked */
	p->buf = CORD_cat(prep, p->buf);
	p->buf_len = (size_t)-1;
}

extern_inline void
proto_buf_prepend_string(proto_t p, const char *prep)
{
	proto_buf_prepend_cord(p, CORD_from_char_star(prep));
	return;
}

extern_inline void
proto_buf_append_cord_lock(proto_t p, CORD app)
{
	SXE_MUTEX_LOCK(&p->mtx);
	proto_buf_append_cord(p, app);
	SXE_MUTEX_UNLOCK(&p->mtx);
	return;
}

extern_inline void
proto_buf_append_string_lock(proto_t p, const char *app)
{
	SXE_MUTEX_LOCK(&p->mtx);
	proto_buf_append_string(p, app);
	SXE_MUTEX_UNLOCK(&p->mtx);
	return;
}

extern_inline void
proto_buf_prepend_cord_lock(proto_t p, CORD prep)
{
	SXE_MUTEX_LOCK(&p->mtx);
	proto_buf_prepend_cord(p, prep);
	SXE_MUTEX_UNLOCK(&p->mtx);
	return;
}

extern_inline void
proto_buf_prepend_string_lock(proto_t p, const char *prep)
{
	SXE_MUTEX_LOCK(&p->mtx);
	proto_buf_prepend_string(p, prep);
	SXE_MUTEX_UNLOCK(&p->mtx);
	return;
}


/* all known protocol implementations */
#include "proto-scscp.h"

#include "modules.h"

#endif	/* INCLUDED_protos_h */

/* protos.h ends here */
