#!/usr/bin/env python

import libxml2
import socket
import select
import re
import asyncore


## dummy node
pysh_dummy = libxml2.newNode("dummy");
# the namespace bind to the node dummy
## in case of interactions with other SCSCP servers, replace the line below
## by this one in comment
## pysh_ns = pysh_dummy.newNs("http://www.openmath.org/OpenMath", None);
pysh_ns = pysh_dummy.newNs("http://www.openmath.org/OpenMath", "OM");

def xmlify(arg):
    if type(arg).__name__ == "int" or type(arg).__name__ == "long":
           return omi(arg);
    elif type(arg).__name__ == "str":
           return omstr(arg);
    elif type(arg).__name__=="float":
           return omf(arg);
    elif type(arg).__name__=="bool":
        if arg==True:
            return oms("logic1", "true");
        elif arg == False:
            return  oms("logic1", "false");
    elif type(arg).__name__ =="complex":
           return omcomplex(arg.real, arg.imag);
    elif type(arg).__name__=="list":
           return omlist(*arg);
    elif type(arg).__name__ == "instance": 
         if arg.__module__ == "libxml2":
             return arg;
         elif arg.Name == "Rational":
             return Py2OM(arg);
         elif arg.Name == "Representative":
             return Py2OM(arg);
         elif arg.Name== "Polynomial":
             return OMPolynomial(arg);
         elif arg.Name== "SquareMatrix":
             return OMSquareMatrix(arg);
         elif arg.Name== "Matrix":
             return OMMatrix(arg);
 
    return arg;


def omi(arg):
    "convert an integer to an openmath integer"
    omi = libxml2.newNode("OMI");
    omi.addChild(libxml2.newText(str(arg)));
    ## lodge our namespace
    omi.setNs(pysh_ns);
    return omi;

def omf(arg):
    "convert a floating point number to an openmath floating point number"
    omf = libxml2.newNode("OMF");
    omf.setProp("dec", str(arg));
    ## lodge our namespace
    omf.setNs(pysh_ns);
    return omf;

def omv(arg):
    "create an openmath OMV"
    omv = libxml2.newNode("OMV");
    omv.setProp("name", str(arg));
    ## lodge our namespace
    omv.setNs(pysh_ns);
    return omv;

def omstr(arg):
    "convert a string to an openmath string"
    omstr = libxml2.newNode("OMSTR");
    omstr.addChild(libxml2.newText(arg));
    ## lodge our namespace
    omstr.setNs(pysh_ns);
    return omstr;
    
def omlist(*arg):
    "ease the writing of openmath lists"
    return oma("list1", "list", *arg);

def omlist1(*arg):
    "ease the writing of openmath lists"
    object=oma("list1", "list", *arg);
    return object.last;

def omcomplex(*arg):
    "ease the writing of openmath complex numbers"
    return oma("complex1", "complex_cartesian", *arg);

def Py2OM(arg):
    "convert basic python and record objects to OM"
    if type(arg).__name__ == "int" or type(arg).__name__ == "long":
           return omi(arg); 
    elif type(arg).__name__ == "str":
           return omstr(arg);
    elif type(arg).__name__=="float":
           return omf(arg);
    elif type(arg).__name__=="bool":
        if arg==True:
            return oms("logic1", "true");
        elif arg == False:
            return  oms("logic1", "false");
    elif type(arg).__name__ =="complex":
           return omcomplex(arg.real, arg.imag);
    elif type(arg).__name__=="list":
           return omlist1(arg);
    elif type(arg).__name__ == "instance": 
         if arg.__module__ == "libxml2":
             return arg;
         elif arg.Name== "Rational":
             return OMRational(arg);
         elif arg.Name== "SquareMatrix": #for order2.matrix
             return OMSquareMatrix(arg); 
         elif arg.Name== "Matrix": #for matrix1.matrix
             return OMMatrix(arg);
         elif arg.Name== "Infinity":
             return OMInfinity(arg);
         elif arg.Name== "Representative":
             return OMResidueClassRepresentative(arg);
         elif arg.Name== "ResidueClassRing":
             return OMResidueClassRing(arg);
         elif arg.Name== "FiniteField":
             return OMFiniteField(arg);
         elif arg.Name== "Setname":
             return OMSetname(arg);
         elif arg.Name== "Polynomial":
             return OMPolynomial(arg);
         elif arg.Name== "PrimitiveElement":
             return OMPrimitiveElement(arg);
         elif arg.Name== "NumberField":
             return OMNumberField(arg);
         elif arg.Name== "Order":
             return OMOrder(arg);
         #to be completed as soon as we add new types
    return arg;

 
def omr(arg):
    "construct an openmath OMR"
    omr = libxml2.newNode("OMR");
    omr.setProp("xref", str(arg));
    ## lodge our namespace
    omr.setNs(pysh_ns);
    return omr;
   
def oms(cdname, sym):
    "to construct an openmath OMS"
    oms = libxml2.newNode("OMS");
    oms.setProp("cd", cdname);
    oms.setProp("name", sym);
    ns = oms.setNs(pysh_ns)
    return oms;

def oma(cdname, sym, *args):
    "to construct an openmath OMA"
    oma = libxml2.newNode("OMA");
    oms = libxml2.newNode("OMS");
    oms.setProp("cd", cdname);
    oms.setProp("name", sym);
    oma.addChild(oms);
    for i in args:
             oma.addChild(xmlify(i));
    ## lodge our namespace
    ns = oma.setNs(pysh_ns)
    return oma;

def omatp(*args):
    "to construct an openmath OMATP"
    omatp = libxml2.newNode("OMATP");
    for i in args:
             omatp.addChild(xmlify(i));
    ## lodge our namespace
    ns = omatp.setNs(pysh_ns)
    return omatp;

def omattr(*args):
    "to construct an openmath OMATTR"
    omattr = libxml2.newNode("OMATTR");
    for i in args:
             omattr.addChild(xmlify(i));
    ## lodge our namespace
    ns = omattr.setNs(pysh_ns)
    return omattr;

def documentify(arg):
    omobj = libxml2.newNode("OMOBJ");
    omobj.addChild(arg);
    doc = libxml2.newDoc("1.0");
    doc.setRootElement(omobj);
    ## lodge our namespace
    # in case of interactions with other SCSCP servers, replace the line below 
    # by this one in comment
    # omobj.newNs("http://www.openmath.org/OpenMath", None);
    omobj.newNs("http://www.openmath.org/OpenMath", "OM");
    omobj.setNs(pysh_ns);
    return doc;

def scscpify(arg):
    return "<?scscp start ?>\n" + arg.serialize() + "<?scscp end ?>\n";


def PyElement(arg):
    "convert basic OM objects back to their native python representation"
    #The openmath element are in a transational block:<OMOBJ>OM object</OMOBJ>
    if arg.name == "OMI":
            cont = arg.getContent();
            result = int(cont);
        
    if arg.name == "OMSTR":
            cont = arg.getContent();
            result = str(cont);
        
    if arg.name == "OMF":
            val= arg.prop("dec");
            result =float(val);
           
    return result;

def PyConvert(arg1):
    "convert OM objects to their python representation or to a record"
    #arg=arg1.children.last.last; 
    #use this to get the procedure_completed part of the transational block
    
    if arg1.children.lsCountNode()<=1:
       arg=arg1.children;

    if arg1.children.lsCountNode()==2:
       arg=arg1.children.last.last;
    
    if arg.lsCountNode() == 0: # the OM objects with only attribute
          if arg.name=="OMF":
              result = PyFloat(arg);
          elif arg.name == "OMS":
               if arg.prop("cd")=="logic1":
                      return PyLogic(arg);
               elif arg.prop("cd")=="nums1":
                      A = Infinity(arg);
                      result = A; 
               elif arg.prop("cd")=="fieldname1" or "ringname1":
                      A = Setname(arg);
                      result = A; 
               # will complete as soon as we implement new types
               else:
                      print "Sorry! this object is not printable yet"
                      return;
          # will complete as soon as we implement new types
          else:
                       print "Sorry! this object is not printable yet"
                       return;

    elif arg.lsCountNode() == 1:

           if arg.name == "OMI" or arg.name == "OMSTR":
                       result = PyElement(arg);
           elif arg.children.prop("cd") == "list1":
                       result = [];
           else:
                       print "Sorry! this object is not printable yet"
                       return;

    elif arg.lsCountNode() == 2:

           if arg.children.prop("cd") == "ringname1":
                       A = ClassRing(arg);
                       result = A; 
           elif arg.children.prop("cd") == "finfield1":
                       result = PrimitiveElement(arg);
           elif arg.children.prop("cd") == "list1":
                       result = ListPy(arg);
           #to be completed as soon as we implement new types
           else:
                       print "Sorry! this object is not printable yet"
                       return;

    elif arg.lsCountNode() == 3:
           if arg.children.prop("cd") == "list1":
                       result = ListPy(arg);
           elif arg.children.prop("cd") == "complex1":
                       result = PyComplex(arg);
           elif arg.children.prop("cd") == "nums1":
                       A = Rational(arg);
                       result = A; 
           elif arg.children.prop("cd") == "polyd1":
                       A = Polynomial(arg);
                       result = A; 
           elif arg.children.prop("cd") == "order1":
               if arg.children.prop("name")=="order":
                       A = Order(arg);
                       result = A;
               elif arg.children.prop("name")=="number_field":
                       A = NumberField(arg);
                       result = A;
           elif arg.children.prop("cd") == "setname2":
                       A = FiniteField(arg);
                       result = A; 
           elif arg.children.prop("cd") == "matrix1":
                       A= Matrix(arg);
                       result = A; 
           elif arg.children.prop("cd") == "order2" and \
                   arg.children.prop("name") == "square_matrix":
                       A= SquareMatrix(arg);
                       result = A;
           elif arg.children.prop("cd") == "integer2":
                       A = ResidueClassRepresentative(arg);
                       result =  A; 
           #to be completed as soon as we implement new types
           else:
                       print "Sorry! this object is not printable yet"
                       return;

    elif arg.lsCountNode() > 3:
           if arg.children.prop("name") == "list":
                       result = ListPy(arg);
              
           else:
                       print "Sorry! this object is not printable yet"
                       return;
    return result;

def PyFloat(arg):
    "specialy to convert OMF to python float"
    return float(arg.prop("dec"));

def ListPy(arg):
    "convert complex OM lists to python lists"
    #list1.list(elt1, elt2, ..., eltn) to [elt1, elt2, ..., eltn] 
    List = [];
    if arg.lsCountNode() == 1 and  arg.children.hasProp("cd"):
                    List = [];
    if arg.lsCountNode() == 2:
       if arg.children.next.name in ["OMI","OMSTR", "OMF"]:
          
                    List.append(PyElement(arg.children.next));
   
       else:
               if arg.last.hasProp("cd"):
                   if arg.last.prop("cd")== "logic1":
                       res = PyLogic(arg.last);
                       List.append(res);
                   elif arg.last.prop("cd")== "ringname1" or "fieldname1":
                       A = Setname(arg.last);
                       List.append( A);  
                   elif arg.last.prop("cd")== "nums1":
                       A = Infinity(arg.last);
                       List.append(A); 
                   #to be completed as soon as we implement new types
                   else:
                       print "Sorry! this object is not printable yet";
                       return;

               elif arg.last.children.prop("cd") == "complex1":
                    List.append(PyComplex(arg.last));
               elif arg.last.children.prop("cd") == "ringname1":
                    A = ClassRing(arg.last);
                    List.append(A); 
               elif arg.last.children.prop("cd") == "finfield1":
                    A = PrimitiveElement(arg.last);
                    List.append(A); 
               elif arg.last.children.prop("name") == "rational":
                    A = Rational(arg.last);
                    List.append(A);
               elif arg.last.children.prop("name") == "class":
                    A = ResidueClassRepresentative(arg.last);
                    List.append(A);
               elif arg.last.children.prop("name") == "GFpn":
                    A = FiniteField(arg.last);
                    List.append(A);
               elif arg.last.children.prop("name") == "order":
                    A = Order(arg.last);
                    List.append(A);
               elif arg.last.children.prop("name") == "matrix":
                    A = Matrix(arg.last);
                    List.append(A);
               elif arg.last.children.prop("name") == "number_field":
                    A = NumberField(arg.last);                    
                    List.append(A);
               elif arg.last.children.prop("name") == "square_matrix":
                    A = SquareMatrix(arg.last);
                    List.append(A);
               elif arg.last.children.prop("name") == "DMP":
                    A = Polynomial(arg.last);
                    List.append(A);
               elif arg.last.children.prop("name") == "list":
                    List.append(ListPy(arg.last));
               #to be completed as soon as we implement new types
               else:
                       print "Sorry! this object is not printable yet";
                       return;
                     
    elif arg.lsCountNode() >= 3:
               A = arg.xpathNewValueTree(); # a little bit tricky 
               B = range(arg.lsCountNode()); # so rewrite A as a new list
               G = [];
               for i in B:
                   G.append(A[i]);
               G.remove(G[0]);
               for i in G:
                     if i.name in ["OMI","OMF","OMSTR"]:
                         List.append(PyElement(i));

                     elif i.name == "OMS":
                            if i.prop("cd")=="logic1":
                                res = PyLogic(i);
                                List.append(res);
                            elif i.prop("cd")== "ringname1" or "fieldname1":
                                List.append(Setname(i)); 
                            elif i.prop("cd")== "nums1":
                                A = Infinity(i);
                                List.append(A); 
                            #to be completed as soon as we implement new types
                            else:
                                print "Sorry! this object is not printable yet";
                                return;

                     else: #in case of i.name=="OMA"
                         
                      if i.children.prop("cd") == "complex1":
                              List.append(PyComplex(i));
                      elif i.children.prop("cd") == "ringname1":
                              List.append(ClassRing(i)); 
                      elif i.children.prop("cd") == "finfield1":
                              List.append(PrimitiveElement(i)); 
                      elif i.children.prop("cd") == "nums1":
                                rec= Rational(i); 
                                List.append(rec);
                      elif i.children.prop("cd") == "integer2":
                                A = ResidueClassRepresentative(i);
                                rec= A; 
                                List.append(rec);
                      elif i.children.prop("cd") == "setname2":
                                rec= FiniteField(i); 
                                List.append(rec);
                      elif i.children.prop("cd") == "order1":
                           if i.children.prop("name")=="order":
                                rec= Order(i);        
                                List.append(rec);
                           elif i.children.prop("name") == "number_field":
                                rec= NumberField(i);    
                                List.append(rec);

                      elif i.children.prop("cd") == "order2" and \
                           i.children.prop("name") == "square_matrix":
                                rec= SquareMatrix(i);                      
                                List.append(rec);
                      elif i.children.prop("cd") == "polyd1": 
                                rec= Polynomial(i);                     
                                List.append(rec);
                      elif i.children.prop("name") == "matrix":
                                rec= Matrix(i);
                                List.append(rec);

                      elif i.children.prop("cd") == "list1":
                                List.append(ListPy(i));
                      #to be completed as soon as we implement new types
                      else:
                           print "Sorry! this object is not printable yet";
                           return;                  
    return List;


# The OM objects with complex structures have no basic python representations. 
# So they will be represented as records.
# First we define the classes, and then the objects will be represented as
# records.
# All the objects created as  records get the method Print to print 
#themselves, the attribute variable Name, and some of them get the 
#attribute variable CD.
# Read the definition of each class to get the different attribute variables.

class Matrix:
      " The class to represent the OM matrix1 matrix as a record"
      def __init__(self, arg):
          # the attribute variables are: CoeffRing, CD, Name, Row, 
          # Column, CoeffList and MatrixRows.
          if arg.children.next.children.next.last.name=="OMS": 
              # for basic coeff rings ie: "Z", "Q", "R", "C" 
              self.CoeffRing= arg.children.next.children.next.last.prop("name");
              #the name of the cd containing the name of the CoeffRing
              self.CD= arg.children.next.children.next.last.prop("cd");
          
          elif arg.children.next.children.next.last.name=="OMA":
              # for complex coeff rings
              if arg.children.next.children.next.last.children.prop("cd")=="ringname1":
                  self.CoeffRing = ClassRing(arg.last.prev.children.next.last);
                  self.CD = self.CoeffRing;
              elif arg.children.next.children.next.last.children.prop("cd")=="setname2":
                  self.CoeffRing = FiniteField(arg.last.prev.children.next.last);
                  self.CD = self.CoeffRing;

              #to be completed (for complex coeff ring)

          self.Row= PyElement(arg.children.next.children.next.next.last);
          self.Column= PyElement(arg.children.next.last.last);
          self.CoeffList = []; #we put all the matrix coeff in this list
          self.Name = "Matrix"
          C = arg.last.xpathNewValueTree();
          K = range(arg.last.lsCountNode());
          # rewrite the list C 
          L = [];
          for i in K:
              L.append(C[i]);
          L.remove(L[0]);
          for i in L:
              if i.name in ["OMI", "OMF", "OMSTR"]:
                  self.CoeffList.append(PyElement(i));
                   
              elif i.name == "OMS":
                                res= i.prop("cd"),  \
                                    i.prop("name");
                                self.CoeffList.append(res);  
              else:
                      
                      if i.children.prop("cd") == "complex1":
                                self.CoeffList.append(PyComplex(i));
                      elif i.children.prop("cd") == "nums1":
                                self.CoeffList.append(Rational(i));
                      elif i.children.prop("cd") == "integer2":
                                self.CoeffList.append(ResidueClassRepresentative(i));
                      elif i.children.prop("cd") == "setname2":
                                self.CoeffList.append(FiniteField(i));
                      elif i.children.prop("cd") == "list1":
                                self.CoeffList.append(ListPy(i));
                      #to be completed as soon as we add new types
          
          # we define the attribute variable MatrixRows: list of matrix rows 
          self.MatrixRows = []; # the elements of this list are the matrix rows
          for i in range(self.Row):
              A = []; #rewrite the CoeffList. We use the idea in matrix1 cd.
              for j in range(self.Column):
                  A.append(self.CoeffList[i*self.Column +j]);

              self.MatrixRows.append(A);

          print "rec(Name:=", self.Name, ":";
          print "\n CoeffRing:=", self.CoeffRing;
          print "\n Row:=", self.Row;
          print "\n Column:=", self.Column;
          print "\n MatrixRows:=", self.MatrixRows;
          print "\n)";
          return None;

      #each object of the class Matrix get the method Print to print itself.
      def Print(self):
          "print OM matrix using a record "
          print "rec(Name:=", self.Name, ":";
          print "\n CoeffRing:=", self.CoeffRing;
          print "\n Row:=", self.Row;
          print "\n Column:=", self.Column;
          print "\n MatrixRows:=", self.MatrixRows;
          print "\n)";
          return;

# This function will be used in Py2OM to convert the given record back to OM
def OMMatrix(arg):
    "Convert back the matrix record to OM matrix"
    # rewrite the MatrixRows: put all the coeff in a list K by adding rows
    # one by one
    K=list();
    B = arg.MatrixRows; # the list of the matrix rows
    nrow = len(B); #the number of rows
    ncol = len(B[1]); # the number of columns
    for i in range(nrow):
        C = B[i];
        for j in range(ncol):
            element =C[j];
            K.append(element);

    a=oma("matrix1", "dense", K[0]);
    K.remove(K[0]);
    for  i in K:
        a.addChild(Py2OM(i));  # a little bit tricky here 
    if type(arg.CoeffRing).__name__ == "str": 
          # for basic coeff ring ie "Z", "Q", "R", "C" 
          return oma("matrix1", "matrix", oma("matrix1","matrix_domain", \
                 oma("matrix1", "entry_domain", oms(arg.CD, arg.CoeffRing)), \
                 oma("matrix1", "row_dimension", arg.Row), \
                 oma("matrix1", "column_dimension", arg.Column)),a);

    elif type(arg.CoeffRing).__name__ == "instance": 
         #for complex coeff rings
         return  oma("matrix1", "matrix", oma("matrix1","matrix_domain", \
                 oma("matrix1", "entry_domain", Py2OM(arg.CD)), \
                 oma("matrix1", "row_dimension", arg.Row), \
                 oma("matrix1", "column_dimension", arg.Column)),a);


# This is used only  for order2.square_matrix
class SquareMatrix:
      " The class to represent the OM order2 square matrix as a record"
      def __init__(self, arg):
          if arg.children.next.children.next.last.name=="OMS":
              # for basic coeff ring ie "Z", "Q", "R", "C" 
              self.CoeffRing= arg.children.next.children.next.last.prop("name");
              # the cd containing the name of the CoeffRing
              self.CD= arg.children.next.children.next.last.prop("cd");
          
          elif arg.children.next.children.next.last.name=="OMA":
               # for complex coeff rings
              if arg.children.next.children.next.last.children.prop("cd")=="ringname1":
                  self.CoeffRing =ClassRing(\
                           arg.children.next.children.next.last);
                  self.CD = self.CoeffRing;
              elif arg.children.next.children.next.last.children.prop("cd")=="setname2":
                  self.CoeffRing = FiniteField(\
                           arg.children.next.children.next.last);
                  self.CD = self.CoeffRing;
              # to be completed as soon as we add new sets
          self.Dimension= PyElement(arg.children.next.children.next.next.last);
          #now write the coeff list
          self.CoeffList = [];
          self.Name = "SquareMatrix";
          C = arg.last.xpathNewValueTree();
          K = range(arg.last.lsCountNode());
          L = [];
          for i in K:
              L.append(C[i]);
          L.remove(L[0]);
          for i in L:
              if i.name in ["OMI", "OMF", "OMSTR"]:
                  self.CoeffList.append(PyElement(i));

              elif i.name == "OMS":
                                res= i.prop("cd"),  \
                                    i.prop("name");
                                self.CoeffList.append(res);
              else:

                      if i.children.prop("cd") == "complex1":
                                self.CoeffList.append(PyComplex(i));
                      elif i.children.prop("cd") == "nums1":
                                self.CoeffList.append(PyRational(i));
                      elif i.children.prop("cd") == "integer2":
                                self.CoeffList.append(ResidueClassRepresentative(i));
                      elif i.children.prop("cd") == "setname2":
                                self.CoeffList.append(FiniteField(i));
                      elif i.children.prop("cd") == "list1":
                                self.CoeffList.append(ListPy(i));

          #we define the attribute variable MatrixRows: list of matrix rows 
          self.MatrixRows = []; #the elements of this list are matrix rows
          for i in range(self.Dimension):
              A = [];  #rewrite the CoeffList. We use the idea in matrix1 cd.
              for j in range(self.Dimension):
                  A.append(self.CoeffList[i*self.Dimension +j]);
              self.MatrixRows.append(A);

          print "rec(Name:=", self.Name, ":";
          print "\n CoeffRing:=", self.CoeffRing;
          print "\n Dimension:=", self.Dimension;
          print "\n MatrixRows:=", self.MatrixRows;
          print "\n)";
          return None;

      def Print(self):
          "print OM order2.square_matrix using a record "
          print "rec(Name:=", self.Name, ":"; 
          print "\n CoeffRing:=", self.CoeffRing;
          print "\n Row:=", self.Dimension;
          print "\n MatrixRows:=", self.MatrixRows;
          print "\n)";
          return;

def OMSquareMatrix(arg):
    "Convert back the matrix record to OM matrix"
    # rewrite the MatrixRows: put all the coeff in a list K by adding rows
    # one by one.
    K=list();
    B = arg.MatrixRows; # list of the matrix rows
    nrow = len(B); # the number of rows
    ncol = len(B[1]); # the number of columns
    for i in range(nrow):
        C = B[i];
        for j in range(ncol):
            element =C[j];
            K.append(element);

    a=oma("order2", "entries", K[0]);
    K.remove(K[0]);
    for  i in K:
        a.addChild(Py2OM(i));

    if type(arg.CoeffRing).__name__ == "str":
        # for basic coeff ring ie "Z", "Q", "R", "C"   
        return oma("order2", "square_matrix", oma("order2",\
           "square_matrix_algebra", oma("order2", "matrix_ground_ring", \
           oms(arg.CD, arg.CoeffRing)), oma("order2", "dimension", \
           arg.Dimension)), a);

    elif type(arg.CoeffRing).__name__ == "instance":
         #for complex coeff rings                                              
         return  oma("order2", "square_matrix", oma("order2", \
                 "square_matrix_algebra", oma("order2",\
                 "matrix_ground_ring", Py2OM(arg.CD)), \
                 oma("order2", "dimension", arg.Dimension)),a);


class PrimitiveElement:
      " The class to represent the OM primitive element as a record"
      def __init__(self, arg):
          #The attribute variables are name, Name, CD
          self.size = PyElement(arg.children.next);
          self.Name="PrimitiveElement";
          self.CD = arg.children.prop("cd");
          self.name= arg.children.prop("name");
          print "rec(Name:=", self.Name, ":";
          print "\n Primitive element of the finite field of size:=", self.size;
          print "\n)";
          return None;

      def Print(self):
          "print OM primitive element of a finite field as a Record"
          print "rec(Name:=", self.Name, ":";
          print "\n Primitive element of the finite field of size:=", self.size;
          print "\n)";
          return;

def OMPrimitiveElement(arg):
    "Convert back the primitive element record to OM"
    return oma(arg.CD, arg.name, arg.size);


def PyLogic(arg):
       "convert OM true and false to their correspondant in python"
       cont = arg.prop("name"); 
       if cont == "true":
            value= True; 
       elif cont == "false":
            value= False;
       return value;

class Infinity:
      "The class to represent infinity as a record"
      def __init__(self, arg):
          # the attribute variables are Name, name, CD
          self.name = arg.prop("name");
          self.Name = "Infinity"; 
          self.CD = arg.prop("cd");
          print "rec(Name:=", self.Name, ":";
          print "\n ", self.name;
          print "\n From the CD:=", self.CD;
          print "\n)";
          return None;

      def Print(self):
          print "rec(Name:=", self.Name;
          print "\n ", self.name;
          print "\n From the CD:=", self.CD;
          print "\n)";
          return;

def OMInfinity(arg):
    "Convert back the infinity record to OM infinity"
    return oms(arg.CD, arg.name);


class ResidueClassRepresentative:
      """The class to represent the OM representative of a residue 
        class as a record"""
      def __init__(self, arg):
          # the attribute variables are Name,name, CD, gen, representative
          self.representative = PyElement(arg.children.next);
          self.gen = PyElement(arg.last);
          self.CD = arg.children.prop("cd");
          self.name = arg.children.prop("name");
          self.Name = "Representative"
          print "rec(Name:=", self.Name, ":";
          print "\n ", self.representative;
          print "\n in the residue class ring of integers modulo", self.gen;
          print "\n)";
          return None;

      def Print(self):
          "print OM residue class representative as a Record"
          print "rec(Name:=", self.Name, ":";
          print "\n ", self.representative;
          print "\n in the residue class ring of integers modulo", self.gen;
          print "\n)";
          return;

def OMResidueClassRepresentative(arg):
    "Convert back the record of the representation of the representative"
    "of a residue class to OM object"
    return oma(arg.CD, arg.name, arg.representative, arg.gen);


class ClassRing:
      "The class to represent the OM residue class ring as a record"
      def __init__(self, arg):
          # the attribute variables are gen, CD, Name, name
          self.gen= PyElement(arg.last);
          self.CD = arg.children.prop("cd");
          self.name = arg.children.prop("name");
          self.Name =  "ResidueClassRing";
          print "rec(Name:=", self.Name, ":";
          print "\n Residue class ring of integers modulo", self.gen;
          print "\n)";
          return None;

      def Print(self):
          "print OM residue class ring as a Record"
          print "rec(Name:=", self.Name, ":";
          print "\n Residue class ring of integers modulo", self.gen;
          print "\n)";
          return;

def OMResidueClassRing(arg):
    "Convert back the class ring record to OM object"
    return oma(arg.CD, arg.name, arg.gen);

class Rational:
      "The class to represent the OM rationals as a record"
      def __init__(self, arg):
          # the attribute variables are Numerator, Denominator, CD, Name, name
          self.Numerator = int(arg.children.next.getContent());
          self.Denominator = int(arg.children.next.next.getContent());
          self.CD = arg.children.prop("cd");
          self.name = arg.children.prop("name");
          self.Name =  "Rational";
          print "rec(Name:=", self.Name, ":";
          print "\n Numerator:= %d" %self.Numerator;
          print "\n Denominator:= %d"  %self.Denominator;
          print "\n)"
          return None;

      def Print(self):
          print "rec(Name:=", self.Name, ":";
          print "\n Numerator:= %d" %self.Numerator;
          print "\n Denominator:= %d"  %self.Denominator;
          print "\n)" 
          return;

def OMRational(arg):
    "Convert back the rational record to OM"
    return oma(arg.CD, arg.name, arg.Numerator, arg.Denominator);


class FiniteField:
      "The class to represent the OM finite field as a record"
      def __init__(self, arg):
          # the attribute variables are Name,name, base, exponent, CD
          self.base= PyElement(arg.children.next);
          self.exponent =  PyElement(arg.last);
          self.CD = arg.children.prop("cd");
          self.name = arg.children.prop("name");
          self.Name ="FiniteField"
          print "rec(Name:=", self.Name, ":";
          print "\n Finite field of size", self.base,"^",self.exponent;
          print "\n)";
          return None;

      def Print(self):
          print "rec(Name:=", self.Name, ":";
          print "\n Finite field of size", self.base,"^",self.exponent;
          print "\n)";
          return;

def OMFiniteField(arg):
    "Convert back the finite field record to OM"
    return oma(arg.CD, arg.name, arg.base, arg.exponent); 

class Setname:
      "The class to represent OM sets name as a record "
      def __init__(self, arg):
          # the attribute variables are Name, CD, name
          self.name = arg.prop("name");
          self.CD = arg.prop("cd");
          self.Name = "Setname";
          print "rec(Name:=", self.Name, ":";
          print "\n ", self.name;
          print "\n From the CD:=", self.CD;
          print "\n)";
          return None;

      def Print(self):
          print "rec(Name:=", self.Name, ":";
          print "\n ", self.name;
          print "\n From the CD:=", self.CD;
          print "\n)";
          return;

def OMSetname(arg):
    "Convert back the setname record to OM"
    return oms(arg.CD, arg.name);

class Order:
      " The class to represent number fields order as a record"
      def __init__(self, arg):
          # the attribute variables are Name, name, CD, cd, CoeffRing, 
          # DefiningPolynomial
          self.CoeffRing = arg.children.next.prop("name");
          self.CD = arg.children.prop("cd");
          self.name = arg.children.prop("name");
          self.cd = arg.children.next.prop("cd");
          self.Name = "Order";
          self.DefiningPolynomial = Polynomial(arg.last);
          print "rec(Name:=", self.Name, ":";
          print "Equation order over:";
          print "\n CoeffRing:=", self.CoeffRing;
          print "\n DefiningPolynomial:=", self.DefiningPolynomial;
          print "\n)";
          return None;

      def Print(self):
          print "rec(Name:=", self.Name, ":";
          print "Equation order over:";
          print "\n CoeffRing:=", self.CoeffRing;
          print "\n DefiningPolynomial:=", self.DefiningPolynomial;
          print "\n)";
          return;

def OMOrder(arg):
    "Convert back the number field order record to OM"
    return oma(arg.CD, arg.name, oms(arg.cd, arg.CoeffRing),\
                   OMPolynomial(arg.DefiningPolynomial)); 

class NumberField:
      " The class to represent number fields as a record"
      def __init__(self, arg):
          # the attribute variables are Name, name, cd, CD, CoeffRing, 
          # DefinigPolynomial
          self.CoeffRing = arg.children.next.prop("name");
          self.cd = arg.children.next.prop("cd");
          self.CD = arg.children.prop("cd");
          self.name = arg.children.prop("name");
          self.Name = "NumberField";                                  
          self.DefiningPolynomial = Polynomial(arg.last);
          print "rec(Name:=", self.Name, ":";
          print "\n CoeffRing:=", self.CoeffRing;
          print "\n DefiningPolynomial:=", self.DefiningPolynomial;
          print "\n)";
          return None;

      def Print(self):
          print "rec(Name:=", self.Name, ":";
          print "\n CoeffRing:=", self.CoeffRing;
          print "\n DefiningPolynomial:=", self.DefiningPolynomial;
          print "\n)";
          return;

def OMNumberField(arg):
    "Convert back the number field record to OM"
    return oma(arg.CD, arg.name, oms(arg.cd, arg.CoeffRing),\
                   OMPolynomial(arg.DefiningPolynomial)); 

# At this time, this works only with the univariate polynomials
class Polynomial:
      " The class to represent polynomials as a record"
      def __init__(self, arg):
          # the attribute variables are Name, CoeffRing, IndeterminateNumber,
          # Representation, CD
          if arg.children.next.children.next.name=="OMS":
              # for the basic coeff rings: "Z", "Q", "R", "C"
              self.CoeffRing= arg.children.next.children.next.prop("name");
              self.CD= arg.children.next.children.next.prop("cd");
              self.IndeterminateNumber = PyElement(arg.children.\
                                  next.children.next.next);
          elif arg.children.next.children.next.name=="OMA":
              # for the complex coeff rings
              if arg.children.next.children.next.children.prop("cd")=="ringname1":
                  self.CoeffRing = ClassRing(arg.children.next.children.next);
                  self.CD = self.CoeffRing;
              elif arg.children.next.children.next.children.prop("cd")=="setname2":
                  self.CoeffRing = FiniteField(arg.children.next.children.next);

              #to be completed

          self.Name = "Polynomial";
          A = arg.last.xpathNewValueTree();
          B = range(arg.last.lsCountNode());
          L = [];
          for i in B:
              L.append(A[i]);
          L.remove(L[0]);
          self.Representation=[];
          for node in L:
              a = node.children.next;
              degree= PyElement(node.children.next.next);
              #check for the type of coeff
              if a.name in ["OMI", "OMF"]: 
                        coeff= PyElement(a);
              elif a.children.prop("cd") == "complex1":
                        coeff = PyComplex(a);
              elif a.children.prop("cd") == "nums1":
                        coeff = Rational(a);
              elif a.children.prop("cd") == "integer2":
                        coeff = ResidueClassRepresentative(a);
  
              else:
                        print "Sorry! this object is not printable yet";
                        return;
              self.Representation.append(tuple((coeff,degree)));
              self.IndeterminateNumber = PyElement(arg.children.\
                                  next.children.next.next);
          print "rec(Name:=", self.Name, ":";
          print "polynomial given as the list of pairs (coeff, degree):";
          print "\n CoeffRing:=", self.CoeffRing;
          print "\n IndeterminateNumber:=", self.IndeterminateNumber;
          print "\n Representation:=", self.Representation;
          print "\n)";
          return None;

      def Print(self):
          print "rec(Name:=", self.Name, ":";
          print "polynomial given as the list of pairs (coeff, degree):";
          print "\n CoeffRing:=", self.CoeffRing;
          print "\n IndeterminateNumber:=", self.IndeterminateNumber;
          print "\n Representation:=", self.Representation;
          print "\n)";
          return;

def OMPolynomial(arg):
    "Convert back the polynomial record to OM"
    M = arg.Representation;
    # will rewrite M as a python list
    K=list();
    j=0;
    while j < len(M):
       K.append(M[j]);
       j=j+1;
    # K is a list of pairs (coef, degree) ie K[i]=(coef, degree)
    # l will be list(K[i]) ==> to get this: [coef, degree]
    # so l[0] is the coef a_i and l[1] is the degree i
    l = list(K[0]); # which is also [K[0]];
    coef = l[0]; #which is K[0][0]
    expo = l[1]; #which is K[0][1]
    a=oma("polyd1", "term", coef, expo);
    b=oma("polyd1", "SDMP", a);
    # go on with K instead M because of a bug with M using the method remove
    K.remove(K[0]); # cause already get it! 
    for  i in K:  # now add the other pairs (coef, degree)
        l = list(i);
        coef = l[0]; #K[m][0] for an index m
        expo = l[1]; #K[m][1]
        b.addChild(oma("polyd1", "term", Py2OM(coef), expo));
    
    #check now for the type of the coef ring
    if type(arg.CoeffRing).__name__ == "str": 
          # for basic coef ring ie "Z", "Q", "R", "C" 
        return oma("polyd1", "DMP", oma("polyd1", "poly_ring_d", \
              oms(arg.CD, arg.CoeffRing), arg.IndeterminateNumber), b);
    if type(arg.CoeffRing).__name__ == "instance": 
          # for complex coef ring
        return oma("polyd1", "DMP", oma("polyd1", "poly_ring_d", \
                  Py2OM(arg.CD), arg.IndeterminateNumber), b);


def PyComplex(arg):
    "Converte OM complex numbers back to their python representation"
    tab =[];
    for elt in arg:
        if elt.hasProp("dec"):
            cont1 = elt.prop("dec")
            tab.append(float(cont1));

    return complex(tab[0], tab[1]);

# This function is used to ease the writing of the request to the server
# The option used by default is "option_return_object" 
def Request(arg, option="option_return_object"):
    "ease the writing of the request: arg is really the procedure_call part."
    
    if option=="option_return_object":
        return omattr(omatp(oms("scscp1", "call_id"), "123.sylla.symcomp", \
           oms("scscp1", option), " "), oma("scscp1", \
           "procedure_call", arg));

    elif option=="option_return_cookie":
        return omattr(omatp(oms("scscp1", "call_id"), "123.sylla.symcomp", \
           oms("scscp1", "option_return_cookie"), " "), oma("scscp1", \
           "procedure_call", oma("scscp2", "store_session", arg)));

    elif option=="option_return_nothing":
        return omattr(omatp(oms("scscp1", "call_id"), "123.sylla.symcomp", \
           oms("scscp1", option), " "), oma("scscp1", \
           "procedure_call",  " ")); 


class connect (asyncore.dispatcher):
    def __init__(self, host, port):
        asyncore.dispatcher.__init__(self);
        self.create_socket(socket.AF_INET, socket.SOCK_STREAM);
        self.connect((host, port));
        self.bufout = "";
        self.bufin = "";
        self.wait_for_reply_p = False;
        connect.handshake(self);
        return None;

    def handle_connect(self):
        pass;
        return;

    def handle_close(self):
        self.close();
        return;

    def readable(self):
        return self.wait_for_reply_p;


    def handle_read(self):
        tmp = self.recv(8192);
        self.bufin += tmp;
        if "<?scscp end ?>" in self.bufin:
           self.wait_for_reply_p = False;
        elif not(self.handshakenp) and \
         "<?scscp version=\"1.3\" ?>" in self.bufin:
           self.handshakenp = True;
           self.wait_for_reply_p = False;
        return;

    def writable(self):
        return (len(self.bufout) > 0);

    def handle_write(self):
        sent = self.send(self.bufout);
        self.bufout = self.bufout[sent:];
        return;
    
    def poll(self, timeout = 20):
        # Use the poll() support added to the select module in Python 2.0
        map = asyncore.socket_map
        self.pollster = asyncore.select.poll()

        nobjs = 0;
        for fd, obj in map.items():
            flags = 0;
            if obj.readable():
               flags |= select.POLLIN | select.POLLPRI;
               nobjs += 1;
            if obj.writable():
               flags |= select.POLLOUT;
               nobjs += 1;
            if flags:
               # Only check for exceptions if object was either readable
               # or writable.
               flags |= select.POLLERR | select.POLLHUP | select.POLLNVAL;
               self.pollster.register(fd, flags);
        try:
            r = self.pollster.poll(timeout);
        except select.error, err:
            if err[0] != EINTR:
               raise;
            r = [];
        for fd, flags in r:
            obj = map.get(fd);
            if obj is None:
               continue;
            asyncore.readwrite(obj, flags);
        return nobjs;
    
    def loop(self, timeout = 20):
        while connect.poll(self, timeout) > 0:
             continue;

    def handshake(self):
        self.bufout = "<?scscp version=\"1.3\" ?>\n"
        self.bufin = "";
        self.wait_for_reply_p = True;
        self.handshakenp = False;
        connect.loop(self);
        ## check if handshake succeeded
        if not(self.handshakenp):
           raise(connect, "Handshake failure");
        print("Handshake succeeded");
        return;

    def compute(self, obj):
        ## set up the buffer
        self.bufout = scscpify(documentify(obj));
        self.bufin = "";
        self.wait_for_reply_p = True;
        ## perform
        connect.loop(self);
        ## find start
        ps = self.bufin.find("<?scscp start ?>");
        if (ps == -1):
           raise(connect, "invalid answer");
        elif (self.bufin[ps+16] == '\n'):
           ps += 17;
        else:
           ps += 16;
        ## find end
        pe = self.bufin.find("<?scscp end ?>");
        if (pe == -1):
           raise(connect, "invalid answer");
        ## get that substring
        doc = libxml2.parseDoc(self.bufin[ps : pe]);
        obj = doc.children;
        print(obj.serialize());
        return obj;

    def compute_intermediate(self, obj):
        "use it in case of computation of intermediate objects"
        ## set up the buffer
        self.bufout = scscpify(documentify(obj));
        self.bufin = "";
        self.wait_for_reply_p = True;
        ## perform
        connect.loop(self);
        ## find start
        ps = self.bufin.find("<?scscp start ?>");
        if (ps == -1):
           raise(connect, "invalid answer");
        elif (self.bufin[ps+16] == '\n'):
           ps += 17;
        else:
           ps += 16;
        ## find end
        pe = self.bufin.find("<?scscp end ?>");
        if (pe == -1):
           raise(connect, "invalid answer");
        ## get that substring
        doc = libxml2.parseDoc(self.bufin[ps : pe]);
        #use this to extract the procedure_terminated part of a computation;
        obj = doc.children.children.last.last;
        print(obj.serialize());
        return obj;
    
    #Apply this function to the GAP objects to extract the 
    #procedure_completed part, and then use it in a kant computation
    def gap2kant(self, obj):
        "to extract the procedure_completed part of a computation from  GAP"
        ## set up the buffer                                               
        self.bufout = scscpify(documentify(obj));
        self.bufin = "";
        self.wait_for_reply_p = True;
        ## perform                                                         
        connect.loop(self);                 
        ## find start                                                         
        ps = self.bufin.find("<?scscp start ?>");
        if (ps == -1):
           raise(connect, "invalid answer");
        elif (self.bufin[ps+16] == '\n'):
           ps += 17;
        else:
           ps += 16;
        ## find end                                                           
        pe = self.bufin.find("<?scscp end ?>");
        if (pe == -1):
           raise(connect, "invalid answer");
        ## get that substring                                                  
        doc = libxml2.parseDoc(self.bufin[ps : pe]);
        obj = doc.children.children.last;
        print(obj.serialize());
        return obj;
